import pandas as pd
from pymongo import MongoClient
from datetime import datetime, timedelta
import schedule
import time
import os

class DailyReportGenerator:
    def __init__(self):
        # Connect to MongoDB
        client = MongoClient(os.environ["MONGO_CLIENT"])
        db = client[os.environ["MONGO_DATABASE"]]
        self.collection = db[os.environ["MONGO_COLLECTION"]]
        

    def get_shift_name(self, timestamp):
        hour = timestamp.hour
        if 6 <= hour < 14:
            return 'Shift A'
        elif 14 <= hour < 22:
            return 'Shift B'
        else:
            return 'Shift C'
        

    def map_packer_name(self, camera_name):
        packer_mapping = {
            'camera_41': 'Packer 1',
            'camera_42': 'Packer 2',
            'camera_44': 'Packer 4',
            'camera_45': 'Packer 5',
            'camera_46': 'Packer 6',
            'camera_47': 'Packer 7'
        }
        return packer_mapping.get(camera_name, 'Unknown')
    

    def get_count(self, start_time, end_time, camera_name):
        query = {
            'timestamp': {'$gte': start_time, '$lte': end_time},
            'cameraName': camera_name
        }
        documents = self.collection.find(query).sort('timestamp', 1)
        first_document = None
        last_document = None

        for document in documents:
            if first_document is None:
                first_document = document
            last_document = document

        if first_document is None or last_document is None:
            return 0

        count_difference = int(last_document['cement_bag_count']) - int(first_document['cement_bag_count'])
        return count_difference    


    def create_excel_report(self):
        data = []
        current_time = datetime.now()
        start_time = datetime(current_time.year, current_time.month, current_time.day - 1, 6, 0, 0)
        end_time = datetime(current_time.year, current_time.month, current_time.day, 6, 0, 0)
        
        for camera_name in ['camera_41', 'camera_42', 'camera_44', 'camera_45', 'camera_46', 'camera_47']:
            shift_name = self.get_shift_name(start_time)
            packer_name = self.map_packer_name(camera_name)
            count = self.get_count(start_time, end_time, camera_name)
            
            data.append({
                'Date': start_time.date(),
                'Shift Name': shift_name,
                'Packer Name': packer_name,
                'Count': count
            })

        df = pd.DataFrame(data)
        writer = pd.ExcelWriter('daily_report.xlsx', engine='openpyxl')
        df.to_excel(writer, index=False, sheet_name='Report')
        writer.save()


    def schedule_report_generation(self):
        schedule.every().day.at('08:00').do(self.create_excel_report)

        while True:
            schedule.run_pending()
            time.sleep(1)


# Create an instance of DailyReportGenerator and start scheduling report generation
report_generator = DailyReportGenerator()
report_generator.schedule_report_generation()
