import pandas as pd
from pymongo import MongoClient
from datetime import datetime, timedelta
import schedule
import time
import os

# Connect to MongoDB
client = MongoClient(os.environ["MONGO_CLIENT"])
db = client[os.environ["MONGO_DATABASE"]]
collection = db[os.environ["MONGO_COLLECTION"]]

# Define function to get shift name
def get_shift_name(timestamp):
    hour = timestamp.hour
    if 6 <= hour < 14:
        return 'Shift A'
    elif 14 <= hour < 22:
        return 'Shift B'
    else:
        return 'Shift C'

# Define function to map packer names
def map_packer_name(camera_name):
    packer_mapping = {
        'camera_41': 'Packer 1',
        'camera_42': 'Packer 2',
        'camera_44': 'Packer 4',
        'camera_45': 'Packer 5',
        'camera_46': 'Packer 6',
        'camera_47': 'Packer 7'
    }
    return packer_mapping.get(camera_name, 'Unknown')

# Retrieve count using query
def get_count(start_time, end_time, camera_name):
    query = {
        'timestamp': {'$gte': start_time, '$lte': end_time},
        'cameraName': camera_name
    }
    count = collection.count_documents(query)
    return count

# Create Excel report
def create_excel_report():
    data = []
    current_time = datetime.now()
    start_time = datetime(current_time.year, current_time.month, current_time.day - 1, 6, 0, 0)
    end_time = datetime(current_time.year, current_time.month, current_time.day, 6, 0, 0)
    
    for camera_name in ['camera_41', 'camera_42', 'camera_44', 'camera_45', 'camera_46', 'camera_47']:
        shift_name = get_shift_name(start_time)
        packer_name = map_packer_name(camera_name)
        count = get_count(start_time, end_time, camera_name)
        
        data.append({
            'Date': start_time.date(),
            'Shift Name': shift_name,
            'Packer Name': packer_name,
            'Count': count
        })

    df = pd.DataFrame(data)
    writer = pd.ExcelWriter('daily_report.xlsx', engine='openpyxl')
    df.to_excel(writer, index=False, sheet_name='Report')
    writer.save()

# Schedule the report generation
schedule.every().day.at('08:00').do(create_excel_report)

# Run the scheduler
while True:
    schedule.run_pending()
    time.sleep(1)
