import pandas as pd
from pymongo import MongoClient
from datetime import datetime, timedelta
import os
from openpyxl import load_workbook
from openpyxl.drawing.image import Image
import yaml
from dotenv import load_dotenv
from collections import defaultdict


load_dotenv(dotenv_path='.env')


class DailyReportGenerator:

    def __init__(self):
        # Connect to MongoDB
        client = MongoClient(os.environ["MONGO_CLIENT"])
        db = client[os.environ["MONGO_DATABASE"]]
        self.collection = db[os.environ["MONGO_COLLECTION"]]

        # Load configuration from config.yml
        with open('config.yml') as config_file:
            self.config = yaml.safe_load(config_file)
            self.template_file = self.config['template_file']
            self.camera_mappings = {
                item['camera_name']: item['packer_name']
                for item in self.config['cameras']
            }


    def get_shift_name(self, timestamp):

        hour = timestamp.hour
        if 6 <= hour < 14:
            return 'Shift A'
        elif 14 <= hour < 22:
            return 'Shift B'
        else:
            return 'Shift C'
        

    def get_packer_name(self, camera_name):
        return self.camera_mappings.get(camera_name, 'Unknown')    


    def get_count(self, start_time, end_time, camera_name):
        query = {
            'timestamp': {'$gte': start_time, '$lte': end_time},
            'cameraName': camera_name
        }
        documents = self.collection.find(query).sort('timestamp', 1)

        count_difference = 0
        first_count = 0
        last_count = 0  

        for i, document in enumerate(documents):
            if i == 0:
                first_count = int(document['cement_bag_count'])
            last_count = int(document['cement_bag_count'])

        count_difference = last_count - first_count

        return count_difference
    

    def create_excel_report(self):

        report_file = 'daily_report.xlsx' 
        

        current_time = datetime.now()

        # Calculate the previous day's date
        previous_day = current_time - timedelta(days=1)
        report_date = previous_day.strftime("%Y-%m-%d")
        report_file = f"daily_report_{report_date}.xlsx"

        wb = load_workbook(self.template_file)
        sheet = wb.active

        # Set report date
        sheet['L6'] = report_date
        serial_number = 1

        # Define the shifts and their respective time ranges
        shifts = {
            'Shift A': (datetime(current_time.year, current_time.month, current_time.day -1 , 6, 0, 0),
                        datetime(current_time.year, current_time.month, current_time.day -1, 14, 0, 0)),
            'Shift B': (datetime(current_time.year, current_time.month, current_time.day -1, 14, 0, 0),
                        datetime(current_time.year, current_time.month, current_time.day -1, 22, 0, 0)),
            'Shift C': (datetime(current_time.year, current_time.month, current_time.day- 1, 22, 0, 0),
                        datetime(current_time.year, current_time.month, current_time.day, 6, 0, 0))
        }

        # Start filling details from row 8
        row = 7
        total_count = 0  # Initialize total count
        packer_counts = defaultdict(int)  # Initialize packer counts dictionary

        # Get the camera_names and corresponding packer_names from config.yml
        camera_names = [item['camera_name'] for item in self.config['cameras']]
        packer_names = [item['packer_name'] for item in self.config['cameras']]

        for camera_name, packer_name in zip(camera_names, packer_names):
            for shift, (start_time, end_time) in shifts.items():
                count = self.get_count(start_time, end_time, camera_name)

                # Fill in the data in respective cells
                sheet[f'A{row}'] = serial_number
                sheet[f'B{row}'] = end_time.date()
                sheet[f'C{row}'] = shift
                sheet[f'D{row}'] = packer_name
                sheet[f'E{row}'] = count

                total_count += count  # Accumulate the count
                packer_counts[packer_name] += count  # Accumulate the count for the packer

                row += 1
                serial_number += 1

        # Fill the total count in cell E26
        sheet['F26'] = total_count

        # Fill the packer counts in column L
        for packer_name, count in packer_counts.items():
            packer_row = 9 + packer_names.index(packer_name) * 3            
            sheet[f'F{packer_row}'] = count

        # Save the report file
        wb.save(report_file)

        # Return the absolute path of the generated report file
        file_path = os.path.abspath(report_file)
        return file_path


# Create an instance of DailyReportGenerator
report_generator = DailyReportGenerator()

# Generate the daily report
report_path = report_generator.create_excel_report()

# Print the path of the generated report
print(f"Generated Daily Report: {report_path}")
