from copy import copy

from sqlalchemy import create_engine, inspect
from sqlalchemy.orm import Session

from scripts.config.app_configurations import DBConf
from scripts.db.psql.models.oee_production_losses import DBModelProductionLosses, LossesStepData
from scripts.logging.logging import logger, logging_config


class QueryLayer:
    def __init__(self, db: Session):
        self.session: Session = db
        self.table = DBModelProductionLosses
        self.echo = logging_config["level"].upper() == "DEBUG"
        self.create_table(self.table.__tablename__)

    def create_table(self, table_name):
        try:
            engine = create_engine(DBConf.ASSISTANT_DB_URI, echo=self.echo)
            if not inspect(engine).has_table(table_name):
                orm_table = self.table
                orm_table.__table__.create(bind=engine, checkfirst=True)
        except Exception as e:
            logger.error(f"Error occurred during start-up: {e}", exc_info=True)

    @property
    def column_asset(self):
        return "asset"

    @property
    def column_asset_model(self):
        return "asset_model"

    @property
    def column_unadjusted_loss_in_time(self):
        return "unadjusted_loss_in_time"

    @property
    def column_booked_loss_in_time(self):
        return "booked_loss_in_time"

    def add_to_table(self, input_data, step_id, date):
        list_of_recs = list()
        try:
            data = LossesStepData(**input_data[0]).dict(exclude_none=True)
            for each in data.get('data_grid'):
                each_record = copy(each)
                each_record.pop('assets', None)
                table_obj = self.table(**each_record, step_id=step_id, date=date)
                table_obj.unadjusted_loss_in_time = data.get('unadjusted_loss_in_time')
                table_obj.booked_loss_in_time = data.get('booked_loss_in_time')
                table_obj.planned_loss_in_minutes = data.get('planned_loss_in_minutes')
                table_obj.available_loss_in_minutes = data.get('available_loss_in_minutes')
                list_of_recs.append(table_obj)
            return list_of_recs
        except TypeError:
            raise
        except Exception as e:
            logger.exception(f"Exception occurred while adding to postgres table {e}")

    def insert_data(self, object_models_list, step_id, date):
        try:
            self.delete_data(step_id, date)
            mappings = self.add_to_table(object_models_list, step_id, date)
            self.session.bulk_save_objects(mappings)
            self.session.commit()
            return True
        except TypeError:
            raise
        except Exception as e:
            logger.exception(e)
            raise

    def delete_data(self, step_id, date):
        try:
            self.session.query(self.table) \
                .filter(self.table.step_id == step_id, self.table.date == date) \
                .delete()
            self.session.commit()
            return True
        except Exception as e:
            logger.error(f"Failed to delete existing data, {e}")
            raise
