from typing import Optional, Dict, List

from scripts.constants.app_constants import DatabaseNames, CollectionNames
from scripts.constants.app_constants import SiteConfCollectionKeys
from scripts.db.mongo.schema import MongoBaseSchema
from scripts.utils.common_utils import CommonUtils
from scripts.utils.mongo_util import MongoCollectionBaseClass


class SiteConfSchema(MongoBaseSchema):
    site_name: Optional[str]
    site_info: Optional[Dict]
    customer_project_id: Optional[str]
    site_id: Optional[str]
    product_encrypted: Optional[bool]
    dept: Optional[List]
    line: Optional[List]
    equipment: Optional[List]


class SiteConf(MongoCollectionBaseClass):
    def __init__(self, mongo_client, project_id=None):
        super().__init__(mongo_client, database=DatabaseNames.ilens_configuration,
                         collection=CollectionNames.site_conf)
        self.com_utils = CommonUtils(project_id=project_id)
        self.project_id = project_id

    @property
    def key_customer_project_id(self):
        return SiteConfCollectionKeys.KEY_CUSTOMER_PROJECT_ID

    @property
    def key_site_id(self):
        return SiteConfCollectionKeys.KEY_SITE_ID

    @property
    def key_site_name(self):
        return SiteConfCollectionKeys.KEY_SITE_NAME

    @property
    def key_process_id(self):
        return SiteConfCollectionKeys.KEY_PROCESS_ID

    def get_all_sites(self, filter_dict=None,
                      sort=None, skip=0, limit=None, site_id=None, **query):
        """
        The following function will give all sites for the given set of
        search parameters as keyword arguments
        :param filter_dict:
        :param sort:
        :param skip:
        :param limit:
        :param query:
        :return:
        """
        if site_id is not None:
            query.update({self.key_site_id: site_id})
        sites = self.find(filter_dict=filter_dict, sort=sort, skip=skip, limit=limit, query=query)
        if sites:
            return list(sites)
        return list()

    def find_site_by_site_name(self, site_name, project_id):
        query_json = {self.key_site_name: site_name, self.key_customer_project_id: project_id}
        response = self.find_one(query=query_json)
        if response:
            return dict(response)
        else:
            return dict()

    def find_site_by_site_id(self, site_id, filter_dict=None, find_condition=dict()):
        if bool(find_condition):
            query = find_condition
        else:
            query = {self.key_site_id: site_id}
        site = self.find_one(query=query, filter_dict=filter_dict)
        if site:
            return dict(site)
        return dict()

    def find_site_by_query(self, query):
        site = self.find(query=query)
        if site:
            return site
        return list()

    def find_sites_by_project(self, project_id):
        sites = self.find(query={self.key_customer_project_id: project_id})
        if sites:
            return list(sites)
        return list()

    def find_accessible_sites(self, project_id, accessible_sites):
        query_dict = {self.key_customer_project_id: project_id, self.key_site_id: {"$in": accessible_sites}}
        sites = self.find(query=query_dict)
        if sites:
            return list(sites)
        return list()

    def delete_one_site(self, site_id):
        if site_id:
            query = {self.key_site_id: site_id}
            return self.delete_one(query=query)
        else:
            return False

    def delete_many_site(self, query=None):
        return self.delete_many(query)

    def update_one_site(self, site_id, data, upsert=False):
        query = {self.key_site_id: site_id}
        return self.update_one(data=data, query=query, upsert=upsert)

    def update_one_process(self, process_id, updated_data):
        """
        The following function will update one tag in
        tags collection based on the given query
        """
        query_dict = {self.key_process_id: process_id}
        return self.update_one(data=updated_data, query=query_dict)

    def update_many_site(self, site_id, data):
        query = {self.key_site_id: site_id}
        response = self.update_many(query=query, data=data)
        if response:
            return list(response)
        else:
            return list()

    def insert_one_site(self, data):
        response = self.insert_one(data=data)
        return response

    def insert_many_sites(self, data):
        response = self.insert_many(data=data)
        if response:
            return list(response)
        else:
            return list()

    def distinct_site_by_key(self, query_key, project_id):
        filter_dict = {self.key_customer_project_id: project_id}
        response = self.distinct(query_key, filter_dict)
        if not response:
            return list()
        return response

    def get_site_data_by_aggregate(self, query: list):
        return list(self.aggregate(pipelines=query))
