import time
from typing import Optional, Dict, List

from scripts.constants.app_constants import DatabaseNames, CollectionNames
from scripts.constants.db_keys import TaskKeys
from scripts.db.mongo.schema import MongoBaseSchema
from scripts.utils.mongo_util import MongoCollectionBaseClass


class TaskInstanceSchema(MongoBaseSchema):
    """
    This is the Schema for the Mongo DB Collection.
    All datastore and general responses will be following the schema.
    """
    task_id: Optional[str]
    logbook_id: Optional[str]
    associated_workflow_id: Optional[str]
    associated_workflow_version: Optional[int]
    task_info_id: Optional[str]
    current_status: Optional[str]
    project_id: Optional[str]
    reference_id: Optional[str]
    task_creation_data: Optional[Dict] = dict()
    meta: Optional[Dict] = {}
    current_stage: Optional[str]
    stages: List[str] = []
    task_category: Optional[str]
    task_description: Optional[str]
    master_details: Optional[Dict] = {}
    task_meta_details: Optional[Dict] = {}


class TaskInstance(MongoCollectionBaseClass):
    def __init__(self, mongo_client, project_id=None):
        super().__init__(mongo_client, database=DatabaseNames.ilens_assistant,
                         collection=CollectionNames.task_instances)
        self.project_id = project_id

    @property
    def key_task_id(self):
        return TaskKeys.KEY_TASK_INSTANCE

    @property
    def key_task_creation_data(self):
        return TaskKeys.KEY_TASK_CREATION_DATA

    @property
    def key_associated_workflow_id(self):
        return TaskKeys.KEY_ASSOCIATED_WORKFLOW_ID

    @property
    def key_workflow_version(self):
        return TaskKeys.KEY_WORKFLOW_VERSION

    @property
    def key_current_status(self):
        return TaskKeys.KEY_CURRENT_STATUS

    def find_by_logbooks(self, logbooks_list):
        query = {"logbook_id": {"$in": logbooks_list}}
        tasks = self.find(query)
        if not tasks:
            return list()
        return tasks

    def find_by_task_id(self, task_id: str):
        query = {self.key_task_id: task_id}
        record = self.find_one(query)
        if not record:
            return None
        return TaskInstanceSchema(**record)

    def update_instance_task(self, task_id, data: dict, upsert=False):
        query = {self.key_task_id: task_id}
        return self.update_one(data=data, query=query, upsert=upsert)

    def find_by_workflow(self, workflow_id_list):
        query = {self.key_associated_workflow_id: {"$in": workflow_id_list}}
        tasks = self.find(query)
        if not tasks:
            return list()
        return tasks

    def update_by_workflow_id(self, workflow_id, workflow_version, from_state, to_state):
        query = {self.key_associated_workflow_id: workflow_id, self.key_workflow_version: workflow_version,
                 self.key_current_status: from_state}
        data = {self.key_current_status: to_state, "meta.completed_at": int(time.time() * 1000)}
        return self.update_one(data=data, query=query, upsert=False)

    def update_by_task_id(self, task_id, from_state, to_state):
        query = {self.key_task_id: task_id,
                 self.key_current_status: from_state}
        data = {self.key_current_status: to_state, "meta.completed_at": int(time.time() * 1000)}
        return self.update_one(data=data, query=query, upsert=False)

    def update_task_creation_by_task_id(self, task_id, property_dict):
        query = {self.key_task_id: task_id}
        # To avoid hierarchy overwrite
        task_creation_update = {f"{self.key_task_creation_data}.{x}": y for x, y in property_dict.items()}
        return self.update_one(data=task_creation_update, query=query, upsert=False)
