from typing import Optional, Dict

from scripts.constants.app_constants import DatabaseNames, CollectionNames
from scripts.constants.db_keys import FormPropsKeys
from scripts.db.mongo.schema import MongoBaseSchema
from scripts.utils.mongo_util import MongoCollectionBaseClass


class FormPropsSchema(MongoBaseSchema):
    """
    This is the Schema for the Mongo DB Collection.
    All datastore and general responses will be following the schema.
    """
    step_id: Optional[str]
    form_info: Optional[Dict]


class FormProps(MongoCollectionBaseClass):
    def __init__(self, mongo_client, project_id=None):
        super().__init__(mongo_client, database=DatabaseNames.ilens_assistant,
                         collection=CollectionNames.form_props)
        self.project_id = project_id

    @property
    def key_step_id(self):
        return FormPropsKeys.KEY_STEP_ID

    @property
    def key_form_info(self):
        return FormPropsKeys.KEY_FORM_INFO

    def add_form_info(self, step_id, form_info):
        json_data = {self.key_step_id: step_id, self.key_form_info: form_info}
        self.insert_one(json_data)
        return True

    def update_form_info(self, step_id, form_info, overwrite=True):
        json_data = {self.key_form_info: form_info} if overwrite else form_info
        query = {self.key_step_id: step_id}
        self.update_one(query, json_data, True)
        return True

    def find_by_id(self, step_id: str):
        query = {self.key_step_id: step_id}
        record = self.find_one(query)
        if not record:
            return FormPropsSchema(**dict())
        return FormPropsSchema(**record)

    def find_step_props(self, step_list):
        records = self.find({self.key_step_id: {"$in": step_list}}, {"step_id": 1, "form_info": 1})
        if not records:
            return dict()
        return {x['step_id']: x["form_info"] for x in records}
