import json
import traceback

from fastapi import APIRouter, UploadFile, File, Form, Depends, HTTPException, Request
from fastapi.responses import FileResponse

from scripts.constants.api import FormEndPoints, StageDataEndPoints
from scripts.core.handlers.stages_data import StagesData
from scripts.core.schemas.response_models import DefaultResponse, DefaultFailureResponse
from scripts.core.schemas.stages_data import TemplateListRequest, FetchTemplate, \
    TemplateTableOptions, TemplateKeyValuePairs, UploadedFileList, DeleteTemplate, DeleteDataFile
from scripts.errors import DuplicateTemplateNameError, BulkUploadError, ColumnsMisMatch, InvalidValueFound
from scripts.logging.logging import logger
from scripts.utils.security_utils.decorators import CookieAuthentication
from scripts.utils.security_utils.project_decorator import MetaInfoCookie, MetaInfoSchema

stages_data_router = APIRouter(tags=["Stage services"], prefix=FormEndPoints.api_stages_data)
get_cookies = MetaInfoCookie()
auth = CookieAuthentication()


@stages_data_router.post(StageDataEndPoints.api_create_template)
async def create_template(template_file: UploadFile = File(...), data: str = Form(...), user_id=Depends(auth),
                          meta: MetaInfoSchema = Depends(get_cookies)):
    try:
        if template_file.content_type not in (
                "application/pdf",
                "text/csv",
                "application/vnd.openxmlformats-officedocument.spreadsheetml.sheet"
        ):
            raise HTTPException(400, detail="Invalid document type")
        stage_data_handler = StagesData(project_id=meta.project_id)
        data = json.loads(data)
        response = await stage_data_handler.create_template(template_file, data, user_id)
        if response:
            return DefaultResponse(status="success", message="Saved successfully", data=response).dict()
    except DuplicateTemplateNameError:
        return DefaultFailureResponse(message="Duplicate name exists")
    except Exception as e:
        tb = traceback.format_exc()
        logger.exception(e)
        logger.exception(tb)
        return DefaultFailureResponse(error=e.args, message="Error Occurred while saving template")


@stages_data_router.post(StageDataEndPoints.api_list_template, dependencies=[Depends(auth)])
async def list_template_info(request_data: TemplateListRequest,
                             meta: MetaInfoSchema = Depends(get_cookies)):
    try:
        stage_data_handler = StagesData(project_id=meta.project_id)
        response = await stage_data_handler.list_template_info(request_data)
        return DefaultResponse(status="success",
                               message="Listed successfully",
                               data=response).dict()
    except Exception as e:
        logger.exception(e)
        logger.exception(traceback.format_exc())
        return DefaultFailureResponse(error=e.args)


@stages_data_router.post(StageDataEndPoints.api_template_table_options, dependencies=[Depends(auth)])
async def list_template_info(request_data: TemplateTableOptions,
                             meta: MetaInfoSchema = Depends(get_cookies)):
    try:
        stage_data_handler = StagesData(project_id=meta.project_id)
        response = await stage_data_handler.list_template_table_options(request_data)
        return DefaultResponse(status="success",
                               message="Fetched successfully",
                               data=response).dict()
    except Exception as e:
        logger.exception(e)
        logger.exception(traceback.format_exc())
        return DefaultFailureResponse(error=e.args, message="Error Occurred while listing templates")


@stages_data_router.post(StageDataEndPoints.api_delete_template, dependencies=[Depends(auth)])
async def delete_template_data(request_data: DeleteTemplate,
                               meta: MetaInfoSchema = Depends(get_cookies)):
    try:
        stage_data_handler = StagesData(project_id=meta.project_id)
        response = await stage_data_handler.delete_template(request_data)
        return DefaultResponse(status="success",
                               message="Template deleted Successfully",
                               data=response).dict()
    except Exception as e:
        tb = traceback.format_exc()
        logger.exception(e)
        logger.exception(tb)
        return DefaultFailureResponse(error=e.args, message="Error Occurred while deleting template")


@stages_data_router.post(StageDataEndPoints.api_fetch_template)
async def fetch_task_info(request_data: FetchTemplate, meta: MetaInfoSchema = Depends(get_cookies)):
    try:
        stage_data_handler = StagesData(project_id=meta.project_id)
        response = stage_data_handler.fetch_instance(request_data)
        return DefaultResponse(status="success", data=response).dict()
    except Exception as e:
        tb = traceback.format_exc()
        logger.exception(e)
        logger.exception(tb)
        return DefaultFailureResponse(error=e.args, message="Error occurred in server while fetching task info")


@stages_data_router.get(StageDataEndPoints.api_download_template)
async def download_template(template_id: str, meta: MetaInfoSchema = Depends(get_cookies)):
    try:
        stage_data_handler = StagesData(project_id=meta.project_id)
        file_path, filename = await stage_data_handler.download_template(template_id)
        if file_path:
            return FileResponse(file_path, filename=filename)
        else:
            raise HTTPException(status_code=404, detail="File Not Found")
    except Exception as e:
        tb = traceback.format_exc()
        logger.exception(e)
        logger.exception(tb)
        return DefaultFailureResponse(error=e.args, message="Error occurred while downloading template")


@stages_data_router.post(StageDataEndPoints.api_upload_data_sheet)
def upload_data_sheet(request: Request, data_file: UploadFile = File(...), data: str = Form(...), user_id=Depends(auth),
                      meta: MetaInfoSchema = Depends(get_cookies)):
    try:
        if data_file.content_type not in (
                "application/pdf",
                "text/csv",
                "application/vnd.openxmlformats-officedocument.spreadsheetml.sheet"
        ):
            raise HTTPException(400, detail="Invalid document type")
        stage_data_handler = StagesData(project_id=meta.project_id)
        data = json.loads(data)
        response = stage_data_handler.save_template_data_file(data_file, data, user_id, meta.project_id,
                                                              request_obj=request)
        return DefaultResponse(status="success",
                               message="Template uploaded Successfully",
                               data=response).dict()
    except BulkUploadError:
        return DefaultFailureResponse(error="File type not supported", message="File type not supported")
    except ColumnsMisMatch:
        return DefaultFailureResponse(error="Columns mismatch", message="Columns mismatch")
    except InvalidValueFound as invalidMessage:
        return DefaultFailureResponse(error=invalidMessage, message="Invalid value found in file")
    except Exception as e:
        tb = traceback.format_exc()
        logger.exception(e)
        logger.exception(tb)
        return DefaultFailureResponse(error=e.args, message="Error Occurred while saving data_file")


@stages_data_router.post(StageDataEndPoints.api_get_templates, dependencies=[Depends(auth)])
async def list_templates_dropdown(request_data: TemplateKeyValuePairs,
                                  meta: MetaInfoSchema = Depends(get_cookies)):
    try:
        stage_data_handler = StagesData(project_id=meta.project_id)
        response = await stage_data_handler.get_template_key_value_pairs(request_data)
        return DefaultResponse(status="success", data=response).dict()
    except Exception as e:
        logger.exception(e)
        logger.exception(traceback.format_exc())
        return DefaultFailureResponse(error=e.args)


@stages_data_router.post(StageDataEndPoints.api_get_file_data_list, dependencies=[Depends(auth)])
async def list_uploaded_files(request_data: UploadedFileList,
                              meta: MetaInfoSchema = Depends(get_cookies)):
    try:
        stage_data_handler = StagesData(project_id=meta.project_id)
        response = await stage_data_handler.list_uploaded_files(request_data)
        return DefaultResponse(status="success",
                               message="Listed successfully",
                               data=response).dict()
    except Exception as e:
        logger.exception(e)
        logger.exception(traceback.format_exc())
        return DefaultFailureResponse(error=e.args, message="Error Occurred while listing uploaded files")


@stages_data_router.get(StageDataEndPoints.api_download_data_file)
async def download_data_file(file_id: str, meta: MetaInfoSchema = Depends(get_cookies)):
    try:
        stage_data_handler = StagesData(project_id=meta.project_id)
        file_path, filename = await stage_data_handler.download_data_file(file_id)
        if file_path:
            return FileResponse(file_path, filename=filename)
        else:
            raise HTTPException(status_code=404, detail="File Not Found")
    except Exception as e:
        tb = traceback.format_exc()
        logger.exception(e)
        logger.exception(tb)
        return DefaultFailureResponse(error=e.args, message="Error occurred while downloading file")


@stages_data_router.post(StageDataEndPoints.api_delete_data_file, dependencies=[Depends(auth)])
async def delete_data_file(request_data: DeleteDataFile,
                           meta: MetaInfoSchema = Depends(get_cookies)):
    try:
        stage_data_handler = StagesData(project_id=meta.project_id)
        response = await stage_data_handler.delete_data_file(request_data)
        return DefaultResponse(status="success",
                               message="File deleted Successfully",
                               data=response).dict()
    except Exception as e:
        tb = traceback.format_exc()
        logger.exception(e)
        logger.exception(tb)
        return DefaultFailureResponse(error=e.args, message="Error Occurred while deleting file")
