import traceback

from fastapi import APIRouter, Depends

from scripts.constants.api import FormEndPoints
from scripts.core.handlers.remarks_handler import RemarkHandler
from scripts.core.schemas.response_models import DefaultResponse, DefaultFailureResponse
from scripts.core.schemas.stages import SaveRemarks, FetchRemarks
from scripts.logging.logging import logger
from scripts.utils.security_utils.decorators import CookieAuthentication
from scripts.utils.security_utils.project_decorator import MetaInfoCookie, MetaInfoSchema

remark_router = APIRouter(tags=["Remark services"], prefix=FormEndPoints.api_remark)
auth = CookieAuthentication()
get_cookies = MetaInfoCookie()


@remark_router.post(FormEndPoints.api_save)
async def save_remarks(request_data: SaveRemarks, user_id=Depends(auth),
                       meta: MetaInfoSchema = Depends(get_cookies)):
    try:
        remark_handler = RemarkHandler(project_id=meta.project_id)
        response = remark_handler.save_remarks_data(request_data, user_id)
        return DefaultResponse(status="success", message="Saved comments successfully",
                               data=response).dict()
    except Exception as e:
        tb = traceback.format_exc()
        logger.exception(e)
        logger.exception(tb)
        return DefaultFailureResponse(error=e.args)


@remark_router.post(FormEndPoints.api_get)
async def fetch_remarks(request_data: FetchRemarks, user_id=Depends(auth),
                        meta: MetaInfoSchema = Depends(get_cookies)):
    try:
        remark_handler = RemarkHandler(project_id=meta.project_id)
        response = remark_handler.fetch_remarks(request_data)
        return DefaultResponse(status="success", message="Fetched comments successfully",
                               data=response).dict()
    except Exception as e:
        tb = traceback.format_exc()
        logger.exception(e)
        logger.exception(tb)
        return DefaultFailureResponse(error=e.args)
