import traceback
from datetime import datetime

import pytz
from fastapi import Request

from scripts.config.app_configurations import DatabaseConstants, EnableAuditing
from scripts.constants import StepCategories, FactorsInTriggerCompletion
from scripts.constants.app_constants import AuditingKeys
from scripts.core.engine.component_manipulation import ComponentManipulation
from scripts.core.engine.stage_navigation import StageNavigation
from scripts.core.handlers.form_handler import FormHandler
from scripts.core.schemas.auditing import UserDataEntryRecord
from scripts.core.schemas.stages import StagesList, GetKDataRequest, TriggerData, MarkTaskCompleteRequest, \
    CopyPropertyValues
from scripts.db import mongo_client, PeriodicData, TaskInstanceData, TaskInstance, \
    StepCollection, Workflow, Constants, \
    LogbookLinkInfo, User
from scripts.db.common_aggregates import CommonAggregates
from scripts.db.mongo.ilens_assistant.collections.form_props import FormProps
from scripts.db.mongo.ilens_assistant.collections.logbook import LogbookInfo
from scripts.db.mongo.ilens_assistant.collections.workflow_permissions import Permissions
from scripts.errors import StepsNotConfigured, ImplementationError
from scripts.logging.logging import logger
from scripts.utils.common_utils import CommonUtils
from scripts.utils.formio_parser import get_field_props
from scripts.utils.mongo_util import MongoCollectionBaseClass
from scripts.utils.mqtt_util import push_notification
from scripts.utils.stage_parser import StageParser


class StageHandler:
    def __init__(self, project_id=None):
        self.stage_conn = TaskInstanceData(mongo_client=mongo_client, project_id=project_id)
        self.logbook_conn = LogbookInfo(mongo_client=mongo_client, project_id=project_id)
        self.workflow_conn = Workflow(mongo_client=mongo_client, project_id=project_id)
        self.tasks_conn = TaskInstance(mongo_client=mongo_client, project_id=project_id)
        self.step_conn = StepCollection(mongo_client=mongo_client, project_id=project_id)
        self.const_conn = Constants(mongo_client=mongo_client)
        self.workflow_permissions = Permissions(mongo_client=mongo_client, project_id=project_id)
        self.periodic_conn = PeriodicData(mongo_client, project_id=project_id)
        self.form_props = FormProps(mongo_client, project_id=project_id)
        self.common_utils = CommonUtils(project_id=project_id)
        self.periodic_job_fails = MongoCollectionBaseClass(mongo_client, "ilens_ebpr", "periodic_job_fails")
        self.periodic_job_fails.project_id = project_id
        self.trigger_col = MongoCollectionBaseClass(mongo_client, DatabaseConstants.ilens_assistant_db, "triggers")
        self.trigger_col.project_id = project_id
        self.common_agg = CommonAggregates()
        self.manipulate_comp = ComponentManipulation(project_id=project_id)
        self.stage_nav_engine = StageNavigation(project_id=project_id)
        self.manipulate_comp = ComponentManipulation(project_id=project_id)
        self.logbook_links = LogbookLinkInfo(mongo_client=mongo_client, project_id=project_id)
        self.user = User(mongo_client=mongo_client)
        self.stage_parser = StageParser(project_id=project_id)
        self.form_handler = FormHandler(project_id)

    def get_stages_list(self, request_data: StagesList, user_id, nav_type):
        try:
            user_role = self.common_utils.get_user_roles_by_project_id(user_id=user_id,
                                                                       project_id=request_data.project_id)
            user_role = user_role[0] if user_role else ""
            task_data = self.tasks_conn.find_by_task_id(task_id=request_data.task_id)

            task_data = task_data.dict() if task_data else dict()
            task_status = task_data.get("task_meta_details", {}).get("task_status")
            task_status = "resumed" if not task_status else task_status
            stages = task_data.get("stages")
            stages_dict = dict()
            steps = list()
            if not bool(stages):
                raise StepsNotConfigured("For the selected Task Stages are not configured")
            stage_data = self.stage_conn.find_data_for_multiple_stages(stages_list=stages)
            stage_status_map, stage_status_map_mobile = {}, {}
            for stage in stage_data:
                stage_id = stage["stage_id"]
                stages_dict.update({stage.get("step_id"): stage_id})
                steps.append(stage.get("step_id"))
                status_value = stage.get("status", False)
                stage_status_map.update({stage.get("step_id"): status_value})
                stage_status_map_mobile.update({stage.get("step_id"): bool(stage.get("step_data"))})
            workflow_permissions = self.workflow_permissions.get_data_by_aggregate(
                self.common_agg.get_workflows_permissions(workflow_id=request_data.workflow_id,
                                                          workflow_version=request_data.workflow_version,
                                                          workflow_status=request_data.task_status,
                                                          user_role=user_role))
            final_dict, permissions_dict = self.stage_nav_engine.get_stages(steps, nav_type, user_role,
                                                                            workflow_permissions, stages_dict,
                                                                            stage_status_map,
                                                                            stage_status_map_mobile,
                                                                            request_data.workflow_id,
                                                                            request_data.workflow_version,
                                                                            mobility=request_data.mobile)

            if all([nav_type == "left", not final_dict.get("left_navigation")]):
                raise PermissionError(f"""1. Access permissions not granted to this task. <br/>  
                                      2. Left navigation stages not added. <br/>  
                                      3. Current task status does not have configured permissions""")
            self.stage_nav_engine.logbook_links(task_data, final_dict)
            if nav_type:
                return_json = dict(stages=final_dict.get(f"{nav_type}_navigation", []), actions=permissions_dict,
                                   task_status=task_status)
                return return_json
            return_json = dict(stages=final_dict, actions=permissions_dict, task_status=task_status)
            return return_json
        except Exception as e:
            logger.error(e.args)
            raise

    @staticmethod
    def get_stage_status(task_data):
        stages_status_map = {}
        end = False
        for each in task_data.get("stages"):
            if task_data.get("current_stage") == each and task_data.get("current_stage") != "completed":
                end = True
            if not end:
                stages_status_map.update({each: True})
            else:
                stages_status_map.update({each: False})
        return stages_status_map

    def add_periodic_stage_data(self, request_data: GetKDataRequest, trigger_method=AuditingKeys.data_published):
        try:
            date = str(self.common_utils.get_time_by_ts(request_data.ts / 1000, request_data.tz).date())
            data_pushed = dict(ts=request_data.ts, values=request_data.values)
            self.periodic_conn.save_and_update_periodic_data(date, request_data.step_id, data_pushed)
            if EnableAuditing.periodic_entry_auditing:
                self.audit_periodic_data(request_data.values, request_data.step_id, request_data.ts,
                                         source=trigger_method)
            return True
        except Exception as e:
            if EnableAuditing.periodic_entry_auditing:
                self.audit_periodic_data(request_data.values, request_data.step_id, request_data.ts, e, trigger_method)
            logger.error(f"Exception while saving record {str(e)}")

    def get_tags(self, step_id, missing_tag):
        try:
            tag_list = list()
            if not missing_tag:
                step = self.form_props.find_by_id(step_id)
                if not bool(step.form_info):
                    return list()
                form_props_dict = get_field_props(step.form_info, "time_associated", "true")
                for element, prop in form_props_dict.items():
                    if "manual_entry" in prop.keys() and prop["manual_entry"] == "true":
                        continue
                    for key, val in prop.items():
                        if key == "tag":
                            tag_list.append(val)
                return list(set(tag_list))
            else:
                record_in_job_fail = self.periodic_job_fails.find_one({"metadata.step_id": step_id})
                if record_in_job_fail:
                    missing_tags = record_in_job_fail.get("missed_tags", list())
                    return missing_tags
                return list()
        except Exception as e:
            logger.error(f"Exception in get_tags {str(e)}")
            raise

    def get_time_list(self, step_id):
        try:
            time_list = list()
            step = self.form_props.find_by_id(step_id)
            if not bool(step.form_info):
                return list()
            form_props_dict = get_field_props(step.form_info, "time_associated", "true")
            for element, prop in form_props_dict.items():
                for key, val in prop.items():
                    if key == "time":
                        time_list.append(val)
            return list(set(time_list))
        except Exception as e:
            logger.error(f"Exception in get_time_list {str(e)}")
            raise

    async def get_tag_and_time_list(self, step_id):
        try:
            tag_times = dict()
            step = self.form_props.find_by_id(step_id)
            if not bool(step.form_info):
                return {}
            form_props_dict = get_field_props(step.form_info, "time_associated", "true")
            for element, prop in form_props_dict.items():
                if "manual_entry" in prop.keys() and prop["manual_entry"] in ["true", "True"]:
                    continue
                if prop["time"] not in tag_times:
                    tag_times[prop["time"]] = []
                if "tag" in prop and "time" in prop and prop["tag"] not in tag_times[prop["time"]]:
                    tag_times[prop["time"]].append(prop["tag"])
            return tag_times
        except Exception as e:
            logger.error(f"Exception in get_tags {str(e)}")
            raise

    async def audit_triggered_data(self, trigger_method, step_id, error=None):
        utc_time = self.common_utils.get_time_now()
        ip_address = self.common_utils.get_ip_of_user()
        try:
            audit_model = UserDataEntryRecord(type=AuditingKeys.periodic,
                                              user_id="machine-triggered",
                                              user_name="machine-triggered",
                                              date_time=utc_time,
                                              ip_address=ip_address,
                                              source=trigger_method,
                                              step_id=step_id,
                                              new_value="failed_to_trigger"
                                              )
            if error:
                audit_model.error_logs = error
                audit_model.action_status = "failed"
            self.common_utils.auditing_with_kafka([audit_model.dict()])
        except Exception as e:
            logger.error(f"Error while auditing in add_triggered_data {str(e)}")

    def audit_periodic_data(self, values, step_id, ts, error=None, source=None):
        utc_time = self.common_utils.get_time_now()
        ip_address = self.common_utils.get_ip_of_user()
        try:
            tag_audits = []
            for k, v in values.items():
                audit_model = UserDataEntryRecord(type=AuditingKeys.periodic,
                                                  user_id=AuditingKeys.data_published,
                                                  user_name=AuditingKeys.data_published,
                                                  date_time=utc_time,
                                                  ip_address=ip_address,
                                                  source=source,
                                                  new_value=v,
                                                  tag=k,
                                                  step_id=step_id,
                                                  tag_time=ts)

                tag_audits.append(audit_model.dict())
                if error:
                    audit_model.action_status = "failed"
                    audit_model.error_logs = error
            if not tag_audits:
                return True
            self.common_utils.auditing_with_kafka(tag_audits)
            logger.info(f"Audited record successfully")
            return True
        except Exception as e:
            logger.error(f"Failed in audit_periodic_data for tags: {e}")

    async def add_triggered_data(self, option, request_data: TriggerData, real_time, restrict_blanks, project_id,
                                 request_obj: Request):
        try:
            logger.info(f"Triggered with step_id: {request_data.step_id}")
            data = self.manipulate_comp.table_component(option=option, real_time=real_time,
                                                        step_id=request_data.step_id, project_id=project_id,
                                                        data=request_data, request_obj=request_obj,
                                                        restrict_blanks=restrict_blanks)
            if not data:
                return True
            logger.info(f"Storing data to periodic data col: {data}")
            data_to_mongo = GetKDataRequest(values=data,
                                            ts=request_data.trigger_time,
                                            tz=request_data.tz,
                                            step_id=request_data.step_id)
            if EnableAuditing.form_non_periodic_auditing:
                self.add_periodic_stage_data(data_to_mongo, request_data.trigger_method)
        except ImplementationError:
            raise
        except Exception as e:
            if EnableAuditing.form_non_periodic_auditing:
                await self.audit_triggered_data(request_data.trigger_method, request_data.step_id, e)
            logger.error(f"Exception in add_triggered_data {str(e)}")
            raise

    async def get_periodic_steps(self, skip, limit):
        try:
            periodic_steps = self.step_conn.find({"step_category": StepCategories.PERIODIC},
                                                 {"step_id": 1, "_id": 0},
                                                 skip=skip, limit=limit)
            return [x["step_id"] for x in periodic_steps]

        except Exception as e:
            logger.error(f"Exception in get_periodic_steps {str(e)}")
            raise

    async def trigger_task_completion(self, factor):
        try:
            if factor not in FactorsInTriggerCompletion.CONSOLIDATED:
                raise NotImplementedError
            all_triggers = self.trigger_col.find(
                query={"actions.trigger_at": factor, "actions.action_type": "auto_complete",
                       "trigger_type": "system_init"})
            for trigger in all_triggers:
                workflow_id = trigger.get("workflow_id")
                workflow_version = trigger.get('workflow_version')
                for action in trigger.get("actions", []):
                    from_state = action.get("from_state")
                    to_state = action.get("to_state")
                    if not all([workflow_id, workflow_version, from_state, to_state]):
                        raise KeyError
                    self.tasks_conn.update_by_workflow_id(workflow_id=workflow_id,
                                                          workflow_version=workflow_version,
                                                          from_state=from_state,
                                                          to_state=to_state)
            return True
        except Exception as e:
            logger.error(f"Exception in trigger_task_completion {str(e)}")
            raise

    async def mark_task_complete(self, request_data: MarkTaskCompleteRequest):
        try:
            self.tasks_conn.update_by_task_id(task_id=request_data.task_id,
                                              from_state=request_data.from_status,
                                              to_state=request_data.to_status)
            if request_data.update_status:
                data = self.stage_conn.find_all_data_by_task_id(request_data.task_id)
                for update_data in data:
                    if "step_data" in update_data:
                        update_data["step_data"]["data"][request_data.update_key] = request_data.update_value
                        self.stage_conn.update_stage_data(update_data["stage_id"], update_data)
            return True
        except Exception as e:
            logger.error(f"Exception in mark_task_complete {str(e)}")
            raise

    async def send_notification_for_roles(self, request_data):
        try:
            task_data = self.tasks_conn.find_by_task_id(request_data.task_id)
            trigger_data = self.trigger_col.find_one({"workflow_id": task_data.associated_workflow_id,
                                                      "workflow_version": task_data.associated_workflow_version})
            user_roles, step_name = [], ""
            if task_data.master_details.get('master_steps'):
                step_id = task_data.master_details.get('master_steps')[0]
                step_name = self.step_conn.fetch_step_name(step_id=step_id)
            master_task_data = self.tasks_conn.find_by_task_id(task_data.master_details.get('master_task_id', ""))
            logbook_name = self.logbook_conn.find_by_id(logbook_id=master_task_data.logbook_id).logbook_name
            if trigger_data.get("actions", []):
                for each in trigger_data.get("actions", []):
                    if each.get("action_type", "") == "notification":
                        user_roles = each.get("user_roles", [])
                        break
            notification = dict(
                type="ilens_assistant",
                message=f"{request_data.message} Step: {step_name}, Logbook : {logbook_name}",
                notification_message="notification Generated Successfully",
                notification_status="success",
                available_at=datetime.now().astimezone(
                    pytz.timezone(request_data.tz)).strftime("%d-%m-%Y %I:%M%p"),
                mark_as_read=False
            )
            try:
                user_data = self.user.find_user_data_with_roles(user_roles, project_id=request_data.project_id)
                for each in user_data:
                    push_notification(notification, each.get("user_id"))
                return True
            except Exception as e:
                notification.update(type="ilens_assistant",
                                    message="notification failed to generate",
                                    notification_message="Failed to send notification",
                                    notification_status="failed")
                logger.error(f"Error while sending notification {e.args}")
                logger.error(traceback.format_exc())
        except Exception as e:
            logger.error(f"Exception in sending notification {str(e)}")
            raise

    async def copy_property_values(self, request_data: CopyPropertyValues, user_id, bg_task, request_obj, db):
        try:
            # task_data = self.tasks_conn.find_by_task_id(task_id=request_data.task_id)
            step_data = self.step_conn.fetch_one_step(step_id=request_data.to_step_id)
            if step_data.get("step_category", "") == StepCategories.NON_PERIODIC:
                request_data.submitted_data.get("data", {}).update(request_data.property_dict)
                self.stage_conn.update_by_task_step_id(request_data.task_id, request_data.to_step_id,
                                                       request_data.submitted_data)
            elif step_data.get("step_category", "") == StepCategories.PERIODIC:
                if not request_data.periodic_date:
                    request_data.periodic_date = datetime.now().strftime("%Y-%m-%d")
                periodic_data = self.periodic_conn.find_by_date_and_step(_date=request_data.periodic_date,
                                                                         step_id=request_data.to_step_id)
                manual_data = periodic_data.manual_data
                manual_data.update(request_data.property_dict)
                self.periodic_conn.update_data_with_date_periodic(_date=request_data.periodic_date,
                                                                  step_id=request_data.to_step_id,
                                                                  manual_data={"manual_data": manual_data})
            return True
        except Exception as e:
            logger.error(f"Exception while copying data from one step to another step{str(e)}")
