from importlib import import_module

import requests
from fastapi import Request

from scripts.config.app_configurations import PathToServices
from scripts.constants.app_constants import SubmitAction, CommonStatusCode
from scripts.core.schemas.forms import CustomActionsModel
from scripts.core.schemas.forms import SaveForm
from scripts.db import TaskInstance, TaskInstanceData, StepCollection, mongo_client, TaskInstanceSchema
from scripts.db.mongo.ilens_assistant.collections.triggers import Trigger
from scripts.errors import RequiredFieldMissing, InternalError
from scripts.logging.logging import logger
from scripts.utils.formio_parser import check_required, parse_component
from scripts.utils.sql_db_utils import TicketEntry, SQLDBUtils
from scripts.utils.stage_parser import StageParser


class SubmitActions:
    def __init__(self, project_id=None):
        self.task_inst_conn = TaskInstance(mongo_client, project_id=project_id)
        self.trigger_conn = Trigger(mongo_client, project_id=project_id)
        self.task_data = TaskInstanceData(mongo_client, project_id=project_id)
        self.step_collection = StepCollection(mongo_client, project_id=project_id)
        self.stage_parser = StageParser(project_id=project_id)

    def get_workflow_details(self, task_id):
        task_details = self.task_inst_conn.find_by_task_id(task_id=task_id)
        if not task_details:
            return None
        workflow_id = task_details.associated_workflow_id
        workflow_version = task_details.associated_workflow_version
        if not workflow_id or not workflow_version:
            return None
        return {"workflow_id": workflow_id, "workflow_version": workflow_version}

    @staticmethod
    def custom_user_actions(custom_action: CustomActionsModel):
        try:
            module_obj = import_module(f"scripts.core.engine.custom_actions.{custom_action.action['action_type']}")
            class_obj = getattr(module_obj, "CustomAction")
            message_exists, message = class_obj(custom_action).trigger_action()
            return message_exists, message
        except InternalError:
            raise
        except Exception as e:
            logger.error(f"Exception occurred in custom_user_actions: {e}")

    def get_trigger_data(self, workflow_id,
                         request_data: SaveForm,
                         workflow_version,
                         user_role,
                         db, request_obj: Request,
                         on_click=None):
        try:
            logger.info("Checking trigger data")
            trigger_data = self.trigger_conn.fetch_by_id(workflow_id=workflow_id,
                                                         workflow_version=workflow_version,
                                                         role=user_role,
                                                         on_click=on_click)
            actions = trigger_data.actions
            task_details: TaskInstanceSchema = self.task_inst_conn.find_by_task_id(task_id=request_data.task_id)
            stages = task_details.stages
            for state in actions:
                if state["action_type"] in ["rest_api", "mark_completed", "send_email", "notification", "create_batch",
                                            "finish_batch"]:
                    custom_action_model = CustomActionsModel(task_details=task_details, action=state,
                                                             submitted_data=request_data.submitted_data,
                                                             stage_id=request_data.stage_id,
                                                             on_click=on_click, tz=request_data.tz,
                                                             date=request_data.date, project_id=request_data.project_id,
                                                             request_obj=request_obj)
                    message_exists, message = self.custom_user_actions(custom_action_model)
                    if message_exists:
                        return message_exists, message
                    continue
                if state["from_state"] == request_data.current_status:
                    current_status = state["to_state"]
                    data = {"current_status": current_status}
                    self.task_inst_conn.update_instance_task(task_id=request_data.task_id, data=data, upsert=False)
                    insert_json = {"task_status": current_status}
                    request_json = {"service_type": 'update',
                                    "data": {"task_id": request_data.task_id, "project_id": request_data.project_id,
                                             "data": insert_json}}
                    try:
                        api_url = f'{PathToServices.AUDIT_PROXY}/task/tracker'
                        headers = \
                            {
                                'login-token': request_obj.headers.get('login-token',
                                                                       request_obj.cookies.get('login-token')),
                                'projectId': request_obj.cookies.get("projectId", request_obj.cookies.get("project_id",
                                                                                                          request_obj.headers.get(
                                                                                                              "projectId"))),
                                'userId': request_obj.cookies.get("user_id",
                                                                  request_obj.cookies.get("userId",
                                                                                          request_obj.headers.get(
                                                                                              "userId")))}
                        resp = requests.post(url=api_url, cookies=request_obj.cookies, headers=headers,
                                             json=request_json)
                        logger.debug(f"Resp Code:{resp.status_code}")
                        if resp.status_code in CommonStatusCode.SUCCESS_CODES:
                            response = resp.json()
                            logger.debug(f"Response:{response}")
                    except requests.exceptions.ConnectionError as e:
                        logger.exception(e.args)
                    continue
            if on_click == SubmitAction.save:
                data = self.update_next_stage({}, stages, request_data.stage_id)
                if data:
                    self.task_inst_conn.update_instance_task(task_id=request_data.task_id, data=data, upsert=False)
            return False, False
        except InternalError:
            raise
        except Exception as e:
            logger.error(f"Exception occurred in get_trigger_data: {e}")

    def check_required_fields_filled(self, stages_list: list):
        required_field_missing = False
        submitted_data_all_stages, step_mapping = self.task_data.find_many(stages_list)
        stage_form_components = self.step_collection.find_many(list(step_mapping.values()))
        label_missing = ""
        for stage, stage_data in submitted_data_all_stages.items():
            data = stage_data.get("data", {}) if stage_data else {}
            if not bool(data):
                continue
            for field, val in data.items():
                step = step_mapping[stage]
                if all([step in stage_form_components.keys(), check_required(field, stage_form_components[step]),
                        not bool(val)]):
                    label_missing = parse_component(stage_form_components[step], field, "label")
                    required_field_missing = True
                    break
        if required_field_missing:
            raise RequiredFieldMissing(f"Required field missing: {label_missing}")
        return True

    def update_next_stage(self, data, stages_list, current_stage):
        try:
            left_stages = self.stage_parser.get_stage_parser(stages_list).get("left", [])
            if left_stages[-1] == current_stage:
                data.update({"current_stage": left_stages[left_stages.index(current_stage)]})
            elif current_stage in left_stages:
                data.update({"current_stage": left_stages[left_stages.index(current_stage) + 1]})
        except Exception as e:
            logger.exception(f'Exception occurred in update next stage definition {e.args}')
        return data

    @staticmethod
    def update_task_status(status, db, task_id):
        _tue_ = TicketEntry
        _sbu_ = SQLDBUtils(db=db)
        update_json = {_tue_.column_event_status(): status}
        try:
            _sbu_.update(
                table=_tue_,
                update_json=update_json,
                filters=[{_sbu_.key_filter_expression(): 'eq',
                          _sbu_.key_filter_column(): _tue_.workflow_id,
                          _sbu_.key_filter_value(): task_id
                          }]
            )
            return True
        except Exception as e:
            logger.error(f"Error occurred while updating record {e}", exc_info=_sbu_.enable_traceback())
            raise
