from copy import deepcopy

import numpy as np

from scripts.config.app_configurations import PathToServices, EnableAuditing
from scripts.constants import StepCategories, CustomObjects
from scripts.constants.api import CustomEndPoints
from scripts.constants.app_constants import AuditingKeys
from scripts.constants.date_constants import ui_time_format_data
from scripts.core.engine.data_engine import DataEngine
from scripts.core.handlers.custom_handler import CustomHandler
from scripts.core.schemas.auditing import UserDataEntryRecord
from scripts.core.schemas.custom_models import SaveTableRequest
from scripts.core.schemas.forms import SaveForm
from scripts.db import mongo_client, StepCollection, TaskInstanceData, PeriodicData, User, TriggerStepCollection
from scripts.db.mongo.ilens_assistant.collections.form_props import FormProps
from scripts.db.psql.databases import get_assistant_db
from scripts.errors import RequiredFieldMissing
from scripts.logging.logging import logger
from scripts.utils.common_utils import CommonUtils
from scripts.utils.data_processor import ProcessData
from scripts.utils.ilens_publish_data import KairosWriter

full_date_format = "yyyy-dd-MM HH:mm"


class PeriodicEntry:
    def __init__(self, project_id=None):
        self.steps_conn = StepCollection(mongo_client, project_id=project_id)
        self.task_inst_data = TaskInstanceData(mongo_client, project_id=project_id)
        self.form_props = FormProps(mongo_client, project_id=project_id)
        self.common_utils = CommonUtils(project_id=project_id)
        self.periodic_conn = PeriodicData(mongo_client, project_id=project_id)
        self.kairos_writer = KairosWriter()
        self.data_engine = DataEngine(project_id=project_id)
        self.processor = ProcessData(project_id=project_id)
        self.user_conn = User(mongo_client)
        self.trigger_step = TriggerStepCollection(mongo_client, project_id=project_id)
        self.api_custom_save_to_psql = f"{PathToServices.FORM_MT}{CustomEndPoints.api_save_table}"
        self.custom_handler = CustomHandler()

        self.ref_tag_time_manual_next = {"tag", "time", "time_associated", "manual_entry", "next_day"}
        self.ref_tag_time_manual_previous = {"tag", "time", "time_associated", "manual_entry", "previous_day"}
        self.ref_tag_time_next = {"tag", "time", "time_associated", "next_day"}
        self.ref_tag_time_previous = {"tag", "time", "time_associated", "previous_day"}
        self.ref_tag_time_manual = {"tag", "time", "time_associated", "manual_entry"}
        self.ref_tag_time = {"tag", "time", "time_associated"}
        self.ref_tag_manual_entry = {"tag", "manual_entry"}
        self.ref_manual_entry = {"manual_entry"}

    def save_periodic_data(self, request_data: SaveForm, bg_task, request_obj):
        try:
            stage_data = self.task_inst_data.find_by_id(request_data.stage_id)
            step = self.steps_conn.fetch_one_step(step_id=stage_data.step_id)
            if step["step_category"] not in [StepCategories.PERIODIC, StepCategories.TRIGGER_BASED] or \
                    not request_data.submitted_data or not bool(request_data.submitted_data.get("data")):
                return False
            submitted_data = request_data.submitted_data["data"]
            form_props = self.form_props.find_by_id(stage_data.step_id).form_info
            if not form_props:
                raise RequiredFieldMissing("Form properties not associated to this periodic step")

            if not request_data.date:
                request_data.date = self.common_utils.get_trigger_in_epoch(request_data.triggers,
                                                                           request_data.submitted_data,
                                                                           form_props)
                if not request_data.date:
                    raise RequiredFieldMissing("Date not added in triggers for periodic template")
            datetime_obj = self.common_utils.time_zone_converter(request_data.date, request_data.tz)
            date = str(datetime_obj.date())
            if step["step_category"] == StepCategories.TRIGGER_BASED:
                trigger_props = self.trigger_step.fetch_one_step(step_id=stage_data.step_id, date=date)
                form_props = trigger_props["form_info"] if trigger_props and trigger_props.get(
                    "form_info") else form_props

            next_date = self.common_utils.get_next_date(date, "yyyy-MM-dd", 1)
            previous_date = self.common_utils.get_next_date(date, "yyyy-MM-dd", -1)
            next_day_record = self.periodic_conn.find_by_date_and_step(next_date, stage_data.step_id)
            prev_day_record = self.periodic_conn.find_by_date_and_step(previous_date, stage_data.step_id)
            today_record = self.periodic_conn.find_by_date_and_step(date, stage_data.step_id)

            tag_dict, kairos_dict, next_day_dict, only_manual, iot_param, previous_day_dict = \
                self.form_tag_dicts(form_props, date, datetime_obj, request_data, submitted_data, stage_data.step_id,
                                    step.get("replicate_type"), bg_task, request_obj)

            if any([kairos_dict, iot_param]):
                if set(iot_param.keys()).intersection(set(kairos_dict.keys())):
                    kairos_dict = {_time: (iot_param[_time] if _time in iot_param.keys() else tags) for _time, tags in
                                   kairos_dict.items()}
                else:
                    kairos_dict.update(iot_param)
                self.common_utils.publish_data_to_kafka(kairos_dict, request_data.project_id)
            try:
                periodic_data = dict(data=tag_dict, manual_data=only_manual)
                self.periodic_conn.save_and_update_data(_date=date, data=periodic_data, step_id=stage_data.step_id)
                self.update_relative_data(existing_rec=prev_day_record.data,
                                          relative_date=previous_date,
                                          data_dict=previous_day_dict,
                                          step_id=stage_data.step_id)
                self.update_relative_data(existing_rec=next_day_record.data,
                                          relative_date=next_date,
                                          data_dict=next_day_dict,
                                          step_id=stage_data.step_id)
                if EnableAuditing.form_periodic_auditing:
                    bg_task.add_task(self.form_audit_model, today_record, submitted_data, form_props, date, next_date,
                                     request_data,
                                     stage_data.step_id,
                                     next_day_record.data, next_day_record.manual_data)
            except Exception as e:
                logger.error(f"Failed to update periodic data in mongo, {e}")
                if EnableAuditing.form_periodic_auditing:
                    bg_task.add_task(self.form_audit_model, today_record, submitted_data, form_props, date, next_date,
                                     request_data,
                                     stage_data.step_id,
                                     next_day_record.data, next_day_record.manual_data, str(e.args))
            return True
        except Exception as e:
            logger.error("Failed in save_periodic_data", e)
            raise

    def allow_all_manual_fields(self, tag_dict, next_day_dict, iot_param, project_id):
        merge_dict = deepcopy(tag_dict)
        merge_dict.update(next_day_dict)
        merge_dict.update(iot_param)
        self.common_utils.publish_data_to_kafka(merge_dict, project_id)
        return True

    @staticmethod
    def add_time(next_day_time, next_day_dict, previous_day_time, previous_day_dict, kairos_dict, this_day_time,
                 this_day_dict):
        if next_day_time not in next_day_dict:
            next_day_dict[next_day_time] = {}
        if previous_day_time not in previous_day_dict:
            previous_day_dict[previous_day_time] = {}
        if previous_day_time not in kairos_dict:
            kairos_dict[previous_day_time] = {}
        if this_day_time not in this_day_dict:
            this_day_dict[this_day_time] = {}
        if this_day_time not in kairos_dict:
            kairos_dict[this_day_time] = {}
        if next_day_time not in kairos_dict:
            kairos_dict[next_day_time] = {}

    def save_to_dicts(self, form_props, date, datetime_obj, _type, request_data: SaveForm, data, custom_service_list):
        try:
            this_day_dict, next_day_dict, previous_day_dict = ({} for _ in range(3))
            prop_not_in_data = []
            day_end = datetime_obj.replace(hour=23, minute=59, second=0).timestamp() * 1000
            iot_param = {day_end: {}}
            next_day_time = previous_day_time = this_day_time = ""
            data, custom_service_list, kairos_dict, only_manual_dict = self.custom_handler.custom_data_list(
                data, _type, request_data, form_props, custom_service_list)

            for prop_key, current_props in form_props.items():
                try:
                    if prop_key not in data.keys():
                        prop_not_in_data.append(prop_key)
                        continue
                    time = current_props.get('time')
                    if time:
                        this_day_time = self.common_utils.convert_str_to_ts(date, current_props.get('time'),
                                                                            ui_time_format_data["yyyy-MM-dd HH:mm"],
                                                                            request_data.tz)
                        next_day_time = self.common_utils.add_days_to_epoch(1, this_day_time, request_data.tz)
                        previous_day_time = self.common_utils.add_days_to_epoch(-1, this_day_time, request_data.tz)
                        self.add_time(next_day_time, next_day_dict, previous_day_time, previous_day_dict, kairos_dict,
                                      this_day_time,
                                      this_day_dict)

                    is_manual_data = current_props.get("manual_entry", "false") in ["true"]
                    is_next_day = current_props.get("next_day", "false") in ["true"]
                    is_previous_day = current_props.get("previous_day", "false") in ["true"]
                    tag = current_props.get("tag")

                    configured_properties = set(current_props.keys())
                    if all([self.ref_tag_time_manual_next.issubset(configured_properties), is_manual_data,
                            is_next_day]):
                        next_day_dict[next_day_time][tag] = data[prop_key]
                        kairos_dict[next_day_time][tag] = data[prop_key]

                    elif all([self.ref_tag_time_manual_previous.issubset(configured_properties), is_manual_data,
                              is_previous_day]):
                        previous_day_dict[previous_day_time][tag] = data[prop_key]
                        kairos_dict[previous_day_time][tag] = data[prop_key]

                    elif all([self.ref_tag_time_previous.issubset(configured_properties), is_previous_day]):
                        previous_day_dict[previous_day_time][tag] = data[prop_key]

                    elif all([self.ref_tag_time_next.issubset(configured_properties), is_next_day]):
                        next_day_dict[next_day_time][tag] = data[prop_key]

                    elif all([self.ref_tag_time_manual.issubset(configured_properties), is_manual_data]):
                        this_day_dict[this_day_time][tag] = data[prop_key]
                        kairos_dict[this_day_time][tag] = data[prop_key]

                    elif all([self.ref_tag_time.issubset(configured_properties)]):
                        this_day_dict[this_day_time][tag] = data[prop_key]

                    elif all([self.ref_tag_manual_entry.issubset(configured_properties), is_manual_data]):
                        iot_param[day_end][tag] = data[prop_key]
                        only_manual_dict[prop_key] = data[prop_key]

                    elif all([self.ref_manual_entry.issubset(configured_properties), is_manual_data]):
                        only_manual_dict[prop_key] = data[prop_key]

                except Exception as e:
                    logger.error(e)
                    raise
            return [this_day_dict, kairos_dict, next_day_dict, only_manual_dict, iot_param, previous_day_dict,
                    custom_service_list]
        except Exception as e:
            logger.error(e)
            raise

    def form_tag_dicts(self, form_props, date, datetime_obj, request_data: SaveForm, data, step_id, _type, bg_task,
                       request_obj):
        custom_service_list = []

        if _type in CustomObjects.custom_models_to_list:
            custom_service_list.append(data)

        [this_day_dict, kairos_dict, next_day_dict, only_manual_dict, iot_param, previous_day_dict,
         custom_service_list] = self.save_to_dicts(
            form_props,
            date,
            datetime_obj,
            _type,
            request_data,
            data,
            custom_service_list)
        if custom_service_list:
            self.custom_save(step_id, date, _type, custom_service_list, bg_task, request_data, request_obj)
        return this_day_dict, kairos_dict, next_day_dict, only_manual_dict, iot_param, previous_day_dict

    def update_relative_data(self, existing_rec, relative_date, data_dict, step_id):
        try:
            if not data_dict:
                return
            if not existing_rec:
                self.periodic_conn.save_and_update_data(relative_date, step_id, dict(data=data_dict))
                return
            existing_dict = {x["ts"]: x["values"] for x in existing_rec}
            new_dict = deepcopy(existing_dict)
            for ts, val in data_dict.items():
                if ts not in existing_dict.keys():
                    new_dict[ts] = val
                else:
                    new_dict[ts] = {**existing_dict[ts], **val}
            self.periodic_conn.save_and_update_data(relative_date, step_id, dict(data=new_dict))
            return
        except Exception as e:
            logger.error("Failed in update_next_day_data", e)
            raise

    def form_audit_model(self, today_record, submitted_data, form_props, current_day, next_date, request_data, step_id,
                         next_day_record, next_manual, error=None):
        try:
            audits = list()
            user_rec = self.user_conn.find_user(request_data.user_id)
            user_name = user_rec.get("username")
            df_list, current_manual = self.current_next_df_list(request_data, form_props, current_day,
                                                                next_date,
                                                                next_day_record, today_record)
            utc_time = self.common_utils.get_time_now()
            ip_address = self.common_utils.get_ip_of_user()
            for each_df in df_list:
                for index, row in each_df.iterrows():
                    if not row.get("values"):
                        row["values"] = "-"
                    row = row.replace({np.nan: None})
                    prop = row['prop']
                    old = row["values"] if row["values"] else ""
                    new = submitted_data.get(prop, "") if submitted_data.get(prop, "") else ""
                    are_equal = self.check_equality(old, new)
                    if are_equal or (old in ["", "-"] and new in ["", "-"]):
                        continue
                    tag_time = row["datetime"] if "datetime" in row and row["datetime"] != np.nan else None
                    audit_model = UserDataEntryRecord(type=AuditingKeys.periodic,
                                                      user_id=request_data.user_id,
                                                      user_name=user_name,
                                                      date_time=utc_time,
                                                      ip_address=ip_address,
                                                      source=AuditingKeys.user,
                                                      previous_value=old,
                                                      new_value=new,
                                                      property_name=prop,
                                                      tag=row.get("tag", ""),
                                                      task_id=request_data.task_id,
                                                      step_id=step_id,
                                                      stage_id=request_data.stage_id,
                                                      project_id=request_data.project_id,
                                                      )

                    if tag_time:
                        audit_model.tag_time = tag_time
                    if error:
                        audit_model.action_status = "failed"
                        audit_model.error_logs = error
                    audits.append(audit_model.dict())
            if current_manual or next_manual:
                for each_entry in [current_manual, next_manual]:
                    for key, val in each_entry.items():
                        old = val
                        new = submitted_data.get(key)
                        are_equal = self.check_equality(old, new)
                        if are_equal:
                            continue
                        audit_model = UserDataEntryRecord(type=AuditingKeys.periodic,
                                                          user_id=request_data.user_id,
                                                          user_name=user_name,
                                                          date_time=utc_time,
                                                          ip_address=ip_address,
                                                          source=AuditingKeys.user,
                                                          previous_value=old,
                                                          tag="manual_entered",
                                                          new_value=new,
                                                          property_name=key,
                                                          task_id=request_data.task_id,
                                                          step_id=step_id,
                                                          stage_id=request_data.stage_id,
                                                          project_id=request_data.project_id,
                                                          )
                        audits.append(audit_model.dict())
                        if error:
                            audit_model.action_status = "failed"
                            audit_model.error_logs = error
            if not audits:
                return True
            self.common_utils.auditing_with_kafka(audits)
            logger.info(f"Audited records successfully")
            return True
        except Exception as e:
            logger.error(f"Failed in form_audit_model: {e}")

    @staticmethod
    def check_equality(old, new):
        """workaround_for_floats"""
        if old == new:
            return True
        try:
            if new:
                new = float(new)
        except ValueError:
            pass
        if all([isinstance(old, (int, float)), isinstance(new, (int, float)), old == new]):
            return True
        return False

    def current_next_df_list(self, request_data, form_props, current_day, next_date, next_day_record,
                             today_record):
        try:
            present_df, next_df, form_df, current_manual = self.data_engine.get_current_and_next_df(request_data,
                                                                                                    form_props,
                                                                                                    next_day_record,
                                                                                                    today_record)
            if present_df.empty:
                if "next_day" in form_df:
                    this_day_props = form_df[form_df["next_day"] != "true"]
                else:
                    this_day_props = form_df.copy(deep=True)
                present_df = this_day_props.rename_axis('prop').reset_index()
            present_df["datetime"] = self.processor.add_timestamp_to_df(present_df, current_day, request_data.tz,
                                                                        ui_time_format_data[full_date_format])
            if next_df.empty and "next_day" in form_df:
                next_day_props = form_df[form_df["next_day"] == "true"]
                next_df = next_day_props.rename_axis('prop').reset_index()
                next_df["datetime"] = self.processor.add_timestamp_to_df(next_df, next_date, request_data.tz,
                                                                         ui_time_format_data[full_date_format])
            elif not next_df.empty:
                next_df["datetime"] = self.processor.add_timestamp_to_df(next_df, next_date, request_data.tz,
                                                                         ui_time_format_data[full_date_format])
            df_list = [present_df, next_df] if not next_df.empty else [present_df]
            return df_list, current_manual
        except Exception as e:
            logger.error(f"Failed in current_next_df_list: {e}")

    def custom_save(self, step_id, date, _type, data_list, bg_task, request_data: SaveForm, request_obj):
        request_model = SaveTableRequest(replicate_type=_type,
                                         data_list=data_list,
                                         step_id=step_id,
                                         date=date,
                                         project_id=request_data.project_id,
                                         tz=request_data.tz,
                                         cookies=request_obj.cookies)
        db = next(get_assistant_db())
        bg_task.add_task(self.custom_handler.save_table_to_postgres, request_model, db)
        return True
