"""Engine related to form rendering service
- Page displayed to user based on his role (Save/Submit/Examination buttons displayed)
- Fetches data with form and saves data to the form

> Input (workflow_instance_id, stage_id)
"""
import json
import re
import threading
import time
from copy import deepcopy
from datetime import datetime

import pandas as pd
import requests
from dateutil.relativedelta import relativedelta
from fastapi import Request

from scripts.config.app_configurations import EnableAuditing, PathToServices, BackFill, EnableEvents
from scripts.constants import StepCategories
from scripts.constants.api import FormEndPoints
from scripts.constants.app_constants import AuditingKeys, CommonStatusCode, SubmitAction
from scripts.constants.date_constants import ui_time_format_data
from scripts.core.engine.custom_implementations import CustomImplementations
from scripts.core.engine.data_engine import DataEngine
from scripts.core.engine.periodic_entry import PeriodicEntry
from scripts.core.engine.submit_actions import SubmitActions
from scripts.core.schemas.auditing import UserDataEntryRecord
from scripts.core.schemas.forms import SaveForm
from scripts.core.schemas.stages import TriggerReferenceData
from scripts.db import mongo_client, StepCollection, TaskInstanceData, PeriodicData, TaskInstance, Trigger, User, \
    FormProps, TaskCollection, TaskInstanceDataSchema, TriggerStepCollection, Workflow
from scripts.db.mongo.ilens_assistant.collections.reference_steps import ReferenceStep
from scripts.db.mongo.ilens_assistant.collections.task_info import TaskSchema
from scripts.db.mongo.ilens_configuration.collections.lookup_table import LookupTable
from scripts.db.mongo.ilens_configuration.collections.site_conf import SiteConf
from scripts.errors import RequiredFieldMissing, ImplementationError, InternalError, QuantityGreaterThanException
from scripts.logging.logging import logger
from scripts.utils.common_utils import CommonUtils
from scripts.utils.data_processor import ProcessData
from scripts.utils.formio_parser import get_field_props, get_field_props_by_keys, get_form_component_info


def background(f):
    """
    a threading decorator
    use @background above the function you want to run in the background
    """

    def backgrnd_func(*a, **kw):
        threading.Thread(target=f, args=a, kwargs=kw).start()

    return backgrnd_func


class FormRenderingEngine:
    def __init__(self, project_id=None):
        self.processor = ProcessData(project_id=project_id)
        self.steps_conn = StepCollection(mongo_client, project_id=project_id)
        self.task_inst_data = TaskInstanceData(mongo_client, project_id=project_id)
        self.task_inst = TaskInstance(mongo_client, project_id=project_id)
        self.task_info = TaskCollection(mongo_client, project_id=project_id)
        self.lookup_table = LookupTable(mongo_client, project_id=project_id)
        self.data_engine = DataEngine(project_id=project_id)
        self.user = User(mongo_client)
        self.task_inst_conn = TaskInstance(mongo_client, project_id=project_id)
        self.trigger_conn = Trigger(mongo_client, project_id=project_id)
        self.form_props = FormProps(mongo_client, project_id=project_id)
        self.common_utils = CommonUtils(project_id=project_id)
        self.submit_action = SubmitActions(project_id=project_id)
        self.periodic_conn = PeriodicData(mongo_client, project_id=project_id)
        self.periodic_data_entry = PeriodicEntry(project_id=project_id)
        self.site_conn = SiteConf(mongo_client, project_id=project_id)
        self.reference_step_data_entry = ReferenceStep(mongo_client, project_id=project_id)
        self.trigger_step_conn = TriggerStepCollection(mongo_client=mongo_client, project_id=project_id)
        self.workflow_conn = Workflow(mongo_client, project_id=project_id)
        self.formde_proxy = PathToServices.DATA_ENGINE
        self.backfill_api_path = f'{self.formde_proxy}{FormEndPoints.api_backfill}'
        self.workflow_conn = Workflow(mongo_client, project_id=project_id)
        self.custom_imp = CustomImplementations(project_id=project_id)

    def get_form_based_template(self, step_id):
        try:
            step = self.steps_conn.fetch_one_step(step_id=step_id)
            return step or {}
        except Exception as e:
            logger.error("Failed to fetch template", e)
            raise

    async def save_data_to_stage(self, request_data: SaveForm, bg_task, db, user_id, request_obj: Request):
        try:
            stage_data = self.task_inst_data.find_by_id(stage_id=request_data.stage_id)
            # updating user meta on every form edit
            form_details = self.form_props.find_by_id(step_id=stage_data.step_id)
            current_data = self.task_inst_conn.find_by_task_id(task_id=request_data.task_id)
            if form_details.form_info:
                key = "task_creation_data"
                property_details = get_field_props_by_keys(form_details.form_info, key)
                if property_details:
                    self.stage_filter(request_data=request_data, property_details=property_details, task_instance=current_data)
            current_data.meta.update(self.common_utils.get_user_meta(user_id))
            if EnableEvents.enable_events:
                self.common_utils.trigger_create_event(request_data.dict(), current_data.dict(), user_id, request_obj)
            self.task_inst_conn.update_instance_task(task_id=request_data.task_id, data=dict(meta=current_data.meta),
                                                     upsert=False)
            try:
                if bool(stage_data) and bool(request_data.task_id):
                    is_report_type = True
                    if not bool(form_details.form_info):
                        is_report_type = False
                    if is_report_type:
                        form_props_dict = get_field_props(form_details.form_info, request_data.template_type, "true")
                        if bool(form_props_dict) and bool(current_data):
                            update_json = {stage_data.step_id: request_data.stage_id}
                            task_info_data = self.task_info.find_by_task_id(task_info_id=current_data.task_info_id)
                            task_info_data.previous_stage_details.update(update_json)
                            self.task_info \
                                .update_task(task_info_id=current_data.task_info_id,
                                             data={"previous_stage_details": task_info_data.previous_stage_details})
                self.save_data_to_reference_step(request_data, step_id=stage_data.step_id)

                if self.periodic_data_entry.save_periodic_data(request_data, bg_task, request_obj):
                    self.check_triggers_on_save(request_data, db, request_obj=request_obj)
                    return True
                self.check_triggers_on_save(request_data, db, request_obj=request_obj)
                if request_data.submitted_data and bool(request_data.submitted_data.get("data")):
                    if EnableAuditing.form_non_periodic_auditing:
                        bg_task.add_task(self.audit_submitted_data, request_data, stage_data)
                    self.task_inst_data.update_stage(request_data.stage_id, request_data.submitted_data)
                    self.save_data_in_master_step(request_data=request_data, user_id=user_id)
            except QuantityGreaterThanException:
                raise
            except Exception as e:
                if EnableAuditing.form_non_periodic_auditing:
                    bg_task.add_task(self.audit_submitted_data, request_data, stage_data, error=str(e.args))
        except Exception as e:
            logger.error("Failed to save data to stage", e)
            raise

    def check_triggers_on_save(self, request_data, db, request_obj: Request):
        try:
            role_id = self.common_utils.get_user_roles_by_project_id(user_id=request_data.user_id,
                                                                     project_id=request_data.project_id)
            user_role = role_id[0]
            workflow_details = self.submit_action.get_workflow_details(request_data.task_id)
            if not request_data.date:
                self.common_utils.convert_trigger_date_to_epoch(request_data.triggers, request_data)
            message_exists, message = self.submit_action.get_trigger_data(workflow_id=workflow_details["workflow_id"],
                                                                          workflow_version=workflow_details[
                                                                              "workflow_version"],
                                                                          request_data=request_data,
                                                                          user_role=user_role,
                                                                          on_click=request_data.type,
                                                                          db=db,
                                                                          request_obj=request_obj)
            logger.debug(f"Returned from get_trigger_data: {message_exists, message}")
        except Exception as e:
            logger.exception(f"Exception occurred in check triggers on save definition {e.args}")

    def audit_submitted_data(self, request_data: SaveForm, stage_data: TaskInstanceDataSchema, error=None):
        try:
            user_rec = self.user.find_user(request_data.user_id)
            old_data = stage_data.step_data.get("data", {})
            submitted_data = request_data.submitted_data["data"]
            user_name = user_rec.get("username")
            utc_now = self.common_utils.get_time_now()
            ip_address = self.common_utils.get_ip_of_user()
            audit_list = []
            for prop, val in submitted_data.items():
                old_val = old_data.get(prop, "-")
                if isinstance(old_val, (dict, list)):
                    old_val = json.dumps(old_val)
                if isinstance(val, (dict, list)):
                    val = json.dumps(val)
                if old_val in ["", "-"] and val in ["", "-"]:
                    continue
                audit_model = UserDataEntryRecord(type=AuditingKeys.non_periodic, user_id=request_data.user_id,
                                                  user_name=user_name, ip_address=ip_address, date_time=utc_now,
                                                  source=AuditingKeys.user, previous_value=old_val, new_value=val,
                                                  property_name=prop, task_id=request_data.task_id,
                                                  step_id=stage_data.step_id, stage_id=request_data.stage_id,
                                                  project_id=request_data.project_id)

                if error:
                    audit_model.action_status = "failed"
                    audit_model.error_logs = error
                audit_list.append(audit_model.dict())
            self.common_utils.auditing_with_kafka(audit_list)
            logger.info(f"Audited record successfully")
            return True
        except Exception as e:
            logger.exception(f"Exception occurred while updating the audit data{e}")
            return False

    @staticmethod
    def remove_date_prop(field_props, _data):
        """To avoid overwriting the triggerOnChange components"""
        trigger_prop_dict = {x: y for x, y in field_props.items() if
                             "triggerOnChange" in y.keys() and y["triggerOnChange"] == "true"}
        if not _data:
            return _data, list(trigger_prop_dict.keys())
        for each in trigger_prop_dict.keys():
            _data.pop(each, None)
        return _data, list(trigger_prop_dict.keys())

    def check_trigger_template_and_get_elements(self, step_id, t_date, template_record):
        field_props = {}
        if template_record.get("step_category") == StepCategories.TRIGGER_BASED:
            if temp_record := self.trigger_step_conn.fetch_one_step(step_id=step_id, date=t_date):
                template_record = temp_record
                field_props = temp_record.get("form_info")
        template = template_record.get("field_elements", dict()).get("components", list())

        return template, field_props

    async def recursive_component_manipulation(self, component_list, start_date, end_date, trigger_prop_list, tz):
        for each in component_list:
            if isinstance(each, list):
                await self.recursive_component_manipulation(each, start_date, end_date,
                                                            trigger_prop_list, tz)
            elif each.get("key", "") not in trigger_prop_list:
                _type = list({"columns", "rows", "components"}.intersection(set(each)))
                if _type and isinstance(each[_type[0]], list):
                    await self.recursive_component_manipulation(each[_type[0]], start_date, end_date,
                                                                trigger_prop_list, tz)
                else:
                    continue
            elif "datePicker" in each:
                each.update({"enableMinDateInput": True, "enableMaxDateInput": True})
                each["datePicker"].update(minDate=start_date)
                each["datePicker"].update(maxDate=end_date)
        return component_list

    async def component_date_restriction(self, task_data, component, trigger_prop_list, tz):
        try:
            start_date, end_date = "", ""
            meta_data = task_data.meta
            if meta_data:
                start_date = meta_data.get("created_at", "")

                end_date = meta_data.get("completed_at", "")
                if start_date:
                    start_date = self.common_utils.get_time_by_ts(int(start_date) // 1000, tz, ui_time_format_data[
                        "MM/dd/yyyy"])
                if end_date:
                    end_date = self.common_utils.get_time_by_ts(int(end_date) // 1000, tz, ui_time_format_data[
                        "MM/dd/yyyy"])

            component = await self.recursive_component_manipulation(component, start_date, end_date, trigger_prop_list,
                                                                    tz)
            return component
        except Exception as e:
            logger.error(f"Error occurred in component_date_restriction {e}")
        return component

    async def form_template_with_data(self, request_data: SaveForm, request_obj: Request):
        try:
            base_dict = dict(submitted_data=dict(data=dict()),
                             components=dict())
            stage_data = self.task_inst_data.find_by_id(request_data.stage_id)
            task_data = self.task_inst_conn.find_by_task_id(task_id=request_data.task_id)
            if not stage_data:
                return base_dict
            updated_data = stage_data.step_data.get("data", dict()) if bool(stage_data.step_data) else dict()
            step_id = stage_data.step_id if bool(stage_data) else ""
            template_record = self.get_form_based_template(step_id)
            base_props = self.form_props.find_by_id(step_id).form_info
            if not request_data.date:
                request_data.date = self.common_utils.get_trigger_in_epoch(request_data.triggers,
                                                                           request_data.submitted_data,
                                                                           base_props)
                if not request_data.date:
                    raise RequiredFieldMissing("Date not added in triggers for periodic template")
            datetime_obj = self.common_utils.time_zone_converter(request_data.date, request_data.tz)
            date = str(datetime_obj.date())
            template, field_props = self.check_trigger_template_and_get_elements(step_id, date, template_record)
            if not template:
                return base_dict
            if not field_props:
                # Assign base properties if not trigger step or if trigger step has no data on this date
                field_props = base_props

            # Bind data saved in periodic tables
            machine_data = self.data_engine.get_data_for_date(request_data, step_id, field_props, date, datetime_obj)
            # if not bool(updated_data) and not machine_data: #need to check this,
            if not bool(updated_data):
                updated_data = self.get_previous_submitted_data(task_data=task_data, prop_data=field_props,
                                                                step_data=template_record,
                                                                step_id=step_id,
                                                                template_type=request_data.template_type,
                                                                auto_populate_key=request_data.auto_populate_key,
                                                                stage_data=stage_data,
                                                                category=template_record["step_category"],
                                                                date=date)
                updated_data = {k: v for k, v in updated_data.items() if v}
            if machine_data:
                machine_data.update(updated_data)
                updated_data = deepcopy(machine_data)
            if template_record.get("step_category") == StepCategories.NON_PERIODIC:
                date = None
            # Bind data with IoT params
            if field_props:
                updated_data = self.data_engine.get_iot_param(field_props, updated_data, date, request_data.tz,
                                                              request_obj=request_obj)
                updated_data, trigger_prop_list = self.remove_date_prop(field_props, updated_data)
                template = await self.component_date_restriction(task_data, template, trigger_prop_list,
                                                                 request_data.tz)
            start_date = task_data.meta.get("created_at", "")
            if start_date:
                start_date = self.common_utils.get_time_by_ts(int(start_date) // 1000, request_data.tz, "%Y-%m-%d")
            else:
                start_date = datetime.now().date().strftime("%Y-%m-%d")
            self.populate_tank_level_data(field_props=field_props, step_data=template_record,
                                          request_data=request_data, submitted_data=updated_data,
                                          datetime_obj=datetime_obj, start_date=start_date)
            self.get_data_from_reference_step(field_props=field_props, submitted_data=updated_data,
                                              date=date, task_id=request_data.task_id)
            form_data = dict(submitted_data=dict(data=updated_data),
                             components=template)
            return form_data
        except Exception as e:
            logger.error("Failed to return stage form with data", e)
            raise

    async def back_fill(self, request_data, request_obj: Request):
        task_step_data = self.task_inst_data.find_by_id(request_data.stage_id)
        if not task_step_data:
            logger.error('Record Not Found')
            return f"Data refresh failed "
        task_step_id = task_step_data.step_id
        time_str = self.common_utils.get_iso_format(request_data.date / 1000, timezone=request_data.tz,
                                                    timeformat=ui_time_format_data["dd-MM-yyyy"])
        back_fill_data = {
            "step_ids": [task_step_id],
            "tz": request_data.tz,
            "start_date": time_str,
            "end_date": time_str,
            "interval_in_mins": int(BackFill.interval_in_mins),

        }
        cookies = request_obj.cookies
        headers = {
            'login-token': request_obj.headers.get('login-token', request_obj.cookies.get('login-token')),
            'projectId': request_obj.cookies.get("projectId", request_obj.cookies.get("project_id",
                                                                                      request_obj.headers.get(
                                                                                          "projectId"))),
            'userId': request_obj.cookies.get("userId",
                                              request_obj.cookies.get("user_id", request_obj.headers.get(
                                                  "userId")))}
        resp = requests.post(url=self.backfill_api_path, json=back_fill_data, timeout=30, cookies=cookies,
                             headers=headers)
        logger.info(f"Cookies: {cookies} Headers: {headers}")
        if resp.status_code not in CommonStatusCode.SUCCESS_CODES:
            logger.error('Failed response from back fill api')
            return f"Data refresh failed ,got {resp.status_code}"
        else:
            logger.info('Back fill api successfully executed')
            return "Data refreshed successfully"

    async def submit_data(self, request_data: SaveForm, db, user_id, request_obj: Request, mobile: bool = False):
        try:
            stage_data = self.task_inst_data.find_by_id(stage_id=request_data.stage_id)
            # updating user meta on every form edit
            current_data = self.task_inst_conn.find_by_task_id(task_id=request_data.task_id)
            current_data.meta.update(self.common_utils.get_user_meta(user_id))
            # triggering event
            if EnableEvents.enable_events:
                self.common_utils.trigger_create_event(request_data.dict(), current_data.dict(), user_id, request_obj)
            self.task_inst_conn.update_instance_task(task_id=request_data.task_id, data=dict(meta=current_data.meta),
                                                     upsert=False)
            if request_data.type == SubmitAction.refresh:
                if request_data.triggers:
                    self.common_utils.convert_trigger_date_to_epoch(request_data.triggers, request_data)
                msg = await self.back_fill(request_data, request_obj)
                return msg
            elif request_data.type != SubmitAction.save:
                # Since trigger action check is available on save request
                # and required fields check not required when type is save
                required_field_missing = self.submit_action.check_required_fields_filled(request_data.stages)
                if not required_field_missing:
                    return False
                role_id = self.common_utils.get_user_roles_by_project_id(user_id=request_data.user_id,
                                                                         project_id=request_data.project_id)
                user_role = role_id[0]
                workflow_details = self.submit_action.get_workflow_details(request_data.task_id)
                if not workflow_details:
                    raise ImplementationError("Workflow for this task has been deleted")
                if not request_data.date:
                    self.common_utils.convert_trigger_date_to_epoch(request_data.triggers, request_data)
                message_exists, message = self.submit_action.get_trigger_data(
                    workflow_id=workflow_details["workflow_id"],
                    workflow_version=workflow_details["workflow_version"],
                    user_role=user_role,
                    on_click=request_data.type,
                    db=db, request_obj=request_obj, request_data=request_data)
                logger.debug(f"Returned from get_trigger_data: {message_exists, message}")
                if message_exists:
                    return message
                elif request_data.type == "mark_complete":
                    task_instance_data = self.task_inst_data.find_by_id(stage_id=request_data.stage_id)
                    if task_instance_data:
                        self.task_inst_data.update_stage_data(stage_id=request_data.stage_id,
                                                              data=dict(status=not task_instance_data.status))
                msg = "Form submitted successfully"
                return msg
            return "Form saved successfully"
        except InternalError:
            raise
        except RequiredFieldMissing:
            raise
        except Exception as e:
            logger.error("Failed to return stage form with data", e)
            raise

    def get_previous_submitted_data(self, task_data, prop_data: dict, step_id: str, step_data, template_type,
                                    auto_populate_key: str, stage_data, category: str, date: str):
        final_json = {}
        try:
            if bool(prop_data) and bool(task_data):
                hierarchy_props_dict = get_field_props(prop_data, "hierarchy_populate", "true")
                capacity_props_dict = get_field_props_by_keys(prop_data, "capacity_populate")
                lookup_name = get_field_props_by_keys(prop_data, "lookup_name")
                step_id_data = get_field_props_by_keys(prop_data, ["from_step", "step"])
                step_key_data = get_field_props_by_keys(prop_data, ["from_key", "step_key"])
                capacity_auto_props_dict = get_field_props(prop_data, "capacity_auto_populate", "true")
                previous_keys = get_field_props_by_keys(prop_data, "auto_populate_value")
                field_props_dict = get_field_props(prop_data, auto_populate_key, "true")
                prev_stage_data = TaskInstanceDataSchema()
                task_info_data = TaskSchema()
                is_cross_step = get_field_props(prop_data, template_type, "true")
                if is_cross_step and task_data:
                    task_info_data = self.task_info.find_by_task_id(task_info_id=task_data.task_info_id)
                    if task_info_data.previous_stage_details:
                        prev_stage_data = self.task_inst_data.find_by_id(
                            stage_id=task_info_data.previous_stage_details.get(stage_data.step_id))
                if field_props_dict:
                    final_json.update(self.form_updated_submitted_json(
                        submitted_data=prev_stage_data.step_data.get("data", {}),
                        props_dict=field_props_dict, previous_keys=previous_keys, category=category, step_id=step_id,
                        date=date))
                if step_id_data:
                    final_json.update(
                        self.load_step_data_to_another_step(task_info_data.dict(), step_id_data, step_key_data,
                                                            category, date))
                if any([hierarchy_props_dict, capacity_auto_props_dict, capacity_props_dict]) and \
                        all([task_data, task_data.task_creation_data.get("hierarchy"),
                             task_data.task_creation_data.get("hierarchy", {}).get("site"), step_data,
                             step_data.get("field_elements")]):

                    input_components = get_form_component_info(step_data.get("field_elements"), "input_components")
                    site_id = task_data.task_creation_data.get("hierarchy", {}).get("site")
                    hierarchy_id = task_data.task_creation_data.get("hierarchy", {}).get(
                        task_data.task_creation_data.get("hierarchy", {}).get("hierarchyLevel"))
                    site_data = self.site_conn.find_site_by_site_id(site_id)
                    if bool(site_data) and bool(hierarchy_props_dict):
                        hierarchy_name = self.common_utils.get_hierarchy_name(site_data=site_data,
                                                                              input_data=hierarchy_id)
                        for element in hierarchy_props_dict:
                            if element in input_components:
                                final_json.update({element: hierarchy_name})
                    lookup_dict = {}
                    lookup_list = {}
                    if capacity_auto_props_dict and lookup_name:
                        for k, v in lookup_name.items():
                            if v not in lookup_list:
                                lookup_data = self.get_data_from_lookup(lookup_name=v, lookup_id=hierarchy_id)
                                lookup_list.update({v: lookup_data})
                                lookup_dict.update({k: lookup_data})
                            else:
                                lookup_dict.update({k: lookup_list[v]})
                        for element in capacity_auto_props_dict:
                            if element in input_components and lookup_dict.get(element):
                                final_json.update({element: lookup_dict.get(element, '')})
                    elif capacity_props_dict and lookup_name:
                        for k, v in lookup_name.items():
                            if v not in lookup_list:
                                lookup_data = self.get_data_from_lookup(lookup_name=v,
                                                                        lookup_id=capacity_props_dict.get(k, ''))
                                if lookup_data and k in input_components:
                                    final_json.update({k: lookup_data})

        except Exception as e:
            logger.error("Failed to return stage form with data", e)
        return final_json

    def form_updated_submitted_json(self, submitted_data: dict, props_dict: dict, previous_keys: dict, category: str,
                                    step_id: str, date: str):
        return_json = {}
        try:
            if category in [StepCategories.PERIODIC, StepCategories.TRIGGER_BASED]:
                periodic_data = list(self.periodic_conn.find_data_with_date(step_id=step_id,
                                                                            _date_query={"$lt": date},
                                                                            sort_json={"date": -1}))
                if not periodic_data:
                    return return_json
                manual_entry_data = periodic_data[0].get("manual_data")
                if not manual_entry_data:
                    return return_json
                for k, v in props_dict.items():
                    return_json.update({k: manual_entry_data.get(previous_keys.get(k), manual_entry_data.get(k))})
            else:
                if not submitted_data:
                    return return_json
                for k, v in props_dict.items():
                    return_json.update({k: submitted_data.get(previous_keys.get(k), submitted_data.get(k))})
        except Exception as e:
            logger.exception(f"Exception occurred while auto populating previous task details {e}")
        return return_json

    def load_step_data_to_another_step(self, task_info_data: dict, step_id_data: dict, step_key_data: dict,
                                       category: str, date: str):
        return_json = {}
        stage_details = {}
        try:
            if category in [StepCategories.PERIODIC, StepCategories.TRIGGER_BASED]:
                steps = list(set(step_id_data.values()))
                periodic_data = self.periodic_conn.find_by_date_and_multi_step(step_id_list=steps, _date=date)
                for each in periodic_data:
                    stage_details.update({each["step_id"]: each.get("manual_data", {})})
                for key, step in step_id_data.items():
                    actual_key = step_key_data.get(key)
                    if actual_key:
                        return_json.update({key: stage_details.get(step, dict()).get(actual_key, "")})
                return return_json
            for k, v in step_id_data.items():
                if bool(task_info_data.get("previous_stage_details", {}).get(v)):
                    stage_details.update({k: task_info_data.get("previous_stage_details", {}).get(v)})
            if not bool(list(stage_details.values())):
                return return_json

            stage_data = self.task_inst_data.find_data_for_multiple_stages(stages_list=list(stage_details.values()))
            stage_map_dict = {data.get("stage_id"): data for data in stage_data}
            for k, v in step_key_data.items():
                return_json.update(
                    {k: stage_map_dict.get(stage_details.get(k), {}).get("step_data", {}).get("data", {}).get(v, "")})

        except Exception as e:
            logger.error("Failed to return stage form with data", e)
        return return_json

    def get_data_from_lookup(self, lookup_name, lookup_id):
        response = ""
        try:
            if not lookup_id:
                return response
            lookup_data = self.lookup_table.find_lookup_dict(lookup_name=lookup_name)
            for data in lookup_data.get("lookup_data", []):
                if data.get("lookupdata_id") == lookup_id:
                    response = data.get("lookup_value")
                    return response
        except Exception as e:
            logger.error("Failed to return stage form with data", e)
        return response

    def populate_tank_level_data(self, field_props: dict, step_data: dict, request_data, submitted_data: dict,
                                 datetime_obj, start_date):
        try:
            if not field_props:
                return submitted_data
            parameter_step_id = get_field_props_by_keys(field_props, "parameter_step_id")
            parameter_step_id = list(parameter_step_id.values())[0] if parameter_step_id else None
            if parameter_step_id:
                date = str(self.common_utils.time_zone_converter(request_data.date, request_data.tz).date())
                parameter_record = self.get_form_based_template(parameter_step_id)
                _parameter_field_props = self.form_props.find_by_id(parameter_step_id).form_info
                parameter_template, parameter_field_props = self.check_trigger_template_and_get_elements(
                    parameter_step_id, date, parameter_record)
                tag_tank = get_field_props_by_keys(field_props, "master_tank_tag")
                tag_volume = get_field_props_by_keys(field_props, "master_volume_tag")
                tag_tank = list(tag_tank.values())[0].replace("$",
                                                              "_") if tag_tank else None
                logger.debug(f"Tank Tag details --> {tag_tank}")
                tag_volume = list(tag_volume.values())[0].replace("$",
                                                                  "_") if tag_volume else None
                logger.debug(f"Volume Tag details --> {tag_volume}")
                if not parameter_template or not tag_tank or not tag_volume:
                    return submitted_data
                if not parameter_field_props:
                    parameter_field_props = deepcopy(_parameter_field_props)
                updated_machine_data = self.fetch_machine_data_for_tank_population(datetime_obj=datetime_obj,
                                                                                   request_data=request_data,
                                                                                   parameter_field_props=parameter_field_props,
                                                                                   end_date=date, start_date=start_date,
                                                                                   tag_tank=tag_tank,
                                                                                   parameter_step_id=parameter_step_id,
                                                                                   tag_volume=tag_volume)
                updated_machine_data = {k: float(sum(v)) for k, v in updated_machine_data.items()}
                input_components = [v for k, v in
                                    get_form_component_info(step_data["field_elements"], "components").items() if
                                    v.type == "table"]
                final_keys_list = []
                temp_dict = {}
                if updated_machine_data:
                    for _item in input_components:
                        for index, each_row in enumerate(_item.rows):
                            temp_list = []
                            is_row_valid = True
                            if index == 0:
                                continue
                            for _index, _data in enumerate(each_row):
                                if len(each_row) == 3 and _index == 0:
                                    continue
                                temp_list.append(_data.key)
                                if bool(submitted_data.get(_data.key)):
                                    is_row_valid = False
                            if is_row_valid:
                                final_keys_list.append(temp_list)
                tank_details = list(map(str, list(filter(lambda x: isinstance(x, int), list(submitted_data.values())))))
                logger.debug(f"Tank No detail --> {tank_details}")
                for k, v in updated_machine_data.items():
                    if not k:
                        continue
                    if final_keys_list and len(final_keys_list[0]) == 2:
                        # if str(k) in tank_details:
                        #     continue
                        submitted_data.update({final_keys_list[0][0]: k, final_keys_list[0][-1]: v})
                        final_keys_list.pop(0)

                logger.debug(f"Input Components list --> {temp_dict}")
        except Exception as e:
            logger.error("Failed to load step from parameter_step_id", e)
        return submitted_data

    def save_data_to_reference_step(self, request_data, step_id):
        try:
            reference_dict = {}
            step = self.steps_conn.fetch_one_step(step_id=step_id)
            if not step:
                raise RequiredFieldMissing("step not exists")
            form_props = self.form_props.find_by_id(step_id).form_info
            if not form_props:
                return False
            reference_data = get_field_props(form_props=form_props, search_keys="reference_step", value="true")
            entity_data = get_field_props_by_keys(form_props=form_props, search_keys="entity_name")
            entity_name = list(entity_data.values())[0] if bool(list(entity_data.values())) else ""
            if not request_data.submitted_data or not bool(
                    request_data.submitted_data.get("data")) or not reference_data:
                return False
            if not request_data.date:
                request_data.date = self.common_utils.get_trigger_in_epoch(request_data.triggers,
                                                                           request_data.submitted_data,
                                                                           form_props)
            date = str(self.common_utils.time_zone_converter(request_data.date, request_data.tz).date())
            submitted_data = request_data.submitted_data["data"]
            property_dict = get_field_props_by_keys(form_props=form_props, search_keys="entity_key")
            if step.get("step_category") in [StepCategories.TRIGGER_BASED]:
                self.save_reference_data_for_trigger_steps(step_id=step_id, date=date, submitted_data=submitted_data,
                                                           step_data=step, entity_name=entity_name)
                return True
            for k, v in submitted_data.items():
                reference_dict.update({property_dict.get(k, k): v})
            self.reference_step_data_entry.update_data_with_date(data=reference_dict, _date=date, step_id=step_id,
                                                                 step_category=step.get("step_category"),
                                                                 entity_name=entity_name, task_id=request_data.task_id)
            return True
        except Exception as e:
            logger.error("Failed to save data in reference step", e)
            return False

    def get_data_from_reference_step(self, field_props: dict, submitted_data: dict, date, task_id: str):
        try:
            previous_step_data = {}
            if not field_props:
                return submitted_data
            step_dict = get_field_props_by_keys(field_props, "referred_step")
            step_key_dict = get_field_props_by_keys(field_props, "referred_key")
            date_key_dict = get_field_props(field_props, "referred_date", "false")
            previous_key_date = get_field_props(field_props, "referred_previous_date", "true")
            previous_keys_list = get_field_props_by_keys(field_props, "referred_previous_date")
            task_based_filter = get_field_props_by_keys(field_props, "task_search_enabled")
            non_periodic_search = get_field_props_by_keys(field_props, "non_periodic_search")
            if not step_dict or not step_key_dict:
                return submitted_data
            step_list_by_date = list(set(list(step_dict.values())))
            step_list_not_by_date = []
            for k, _step in date_key_dict.items():
                if k in step_dict:
                    step_list_by_date.remove(step_dict[k])
                    step_list_not_by_date.append(step_dict[k])
            if task_based_filter:
                step_data = self.reference_step_data_entry.find_by_date_and_multi_step(step_id_list=step_list_by_date,
                                                                                       _date=date, task_id=task_id)
            # elif non_periodic_search:
            #     step_data = self.reference_step_data_entry.find_by_multi_step_without_date(step_id_list=step_list_by_date)
            else:
                step_data = self.reference_step_data_entry.find_by_date_and_multi_step(step_id_list=step_list_by_date,
                                                                                       _date=date)

            if previous_key_date:
                _date = (datetime.strptime(date, "%Y-%m-%d") - relativedelta(days=1)).strftime("%Y-%m-%d")
                previous_step_data = self.reference_step_data_entry.find_by_date_and_multi_step(
                    step_id_list=step_list_by_date,
                    _date=_date)
            step_data.update(self.reference_step_data_entry.fetch_data_from_query(
                query={"step_id": {"$in": step_list_not_by_date}}))
            default_step_id = list(step_dict.values())[0] if step_dict else None
            if default_step_id:
                for k, v in step_key_dict.items():
                    if k in previous_keys_list and step_dict[k] in previous_step_data:
                        value = previous_step_data[step_dict[k]].get("data", {}).get(v, "")
                    elif k in step_dict and step_dict[k] in step_data and k not in previous_keys_list:
                        value = step_data[step_dict[k]].get("data", {}).get(v, "")
                    else:
                        value = ""
                    submitted_data.update({k: value})
        except Exception as e:
            logger.error("Failed to fetch data from reference step", e)
        return submitted_data

    async def form_fill_with_reference_data(self, input_data: TriggerReferenceData, entity_name: str,
                                            entity_search: str):
        input_request = SaveForm(**{"type": input_data.type, "tz": input_data.tz, "project_id": input_data.project_id,
                                    "stage_id": input_data.stage_id, "current_status": input_data.current_status,
                                    "user_id": input_data.user_id, "triggers": input_data.triggers,
                                    "task_id": input_data.task_id})
        try:
            submitted_data = {}
            stage_data = self.task_inst_data.find_by_id(input_data.stage_id)
            if not stage_data:
                return submitted_data
            form_props = self.form_props.find_by_id(stage_data.step_id).form_info
            if not form_props:
                return submitted_data
            prop_value = input_data.property_value
            if input_data.field_type.lower() in ["number", "integer", "int"]:
                prop_value = int(prop_value)
            query_dict = {f"data.{input_data.entity_key}": prop_value}
            if entity_search.lower() != "false":
                query_dict.update({"entity_name": entity_name})

            row_unique_data = get_field_props(form_props, "row_unique_key", input_data.row_unique_key)
            step_key_dict = get_field_props_by_keys(form_props, "referred_key")
            records = self.reference_step_data_entry.find_data_from_query(query=query_dict,
                                                                          sort_json={"_id": -1}, find_one=False)
            reference_data = self.common_utils.get_updated_reference_data(records)
            if not bool(reference_data):
                return submitted_data
            for _data in row_unique_data.keys():
                if _data in step_key_dict and step_key_dict[_data] in reference_data["data"]:
                    submitted_data.update({_data: reference_data["data"][step_key_dict[_data]]})
            input_request.submitted_data["data"] = submitted_data
            return input_request.submitted_data
        except Exception as e:
            logger.error("Failed to fill the form with data with AR No", e)
            return {}

    def save_reference_data_for_trigger_steps(self, submitted_data: dict, step_data: dict,
                                              step_id: str, date: str, entity_name: str):
        try:
            form_props = self.trigger_step_conn.fetch_one_step(step_id=step_id, date=date)
            form_props = form_props if bool(form_props) else {}
            property_dict = get_field_props_by_keys(form_props=form_props.get("form_info", {}),
                                                    search_keys="entity_key")
            form_data = deepcopy(submitted_data)
            events_list = list(
                {
                    '_'.join(data.split('_')[-2:])
                    for data in list(submitted_data.keys())
                }
            )
            for event in events_list:
                temp_json = {}
                for k, v in submitted_data.items():
                    if event not in k:
                        continue
                    temp_json.update({property_dict.get(k, k): v})
                    form_data.pop(k)
                submitted_data = deepcopy(form_data)
                self.reference_step_data_entry.update_data_for_trigger_steps(data=temp_json, _date=date,
                                                                             step_id=step_id,
                                                                             step_category=step_data.get(
                                                                                 "step_category"),
                                                                             entity_name=entity_name,
                                                                             event_id=event)
            return True
        except Exception as e:
            logger.error("Failed to save data in reference step for trigger based steps", e)
            return False

    def fetch_machine_data_for_tank_population(self, start_date, end_date, request_data, parameter_step_id,
                                               parameter_field_props, datetime_obj, tag_tank, tag_volume):
        return_dict = {}
        try:
            dates = []
            start_date = (datetime.strptime(start_date, "%Y-%m-%d"))
            end_date = (datetime.strptime(end_date, "%Y-%m-%d"))
            delta = end_date - start_date
            regex_spl = re.compile('[@_!#$%^&*()<>?/\|}{~:]')
            for i in range(delta.days + 1):
                each_day = (start_date + relativedelta(days=i)).strftime("%Y-%m-%d")
                dates.append(each_day)
            machine_data = self.machine_date_for_multiple_dates(request_data=request_data, step_id=parameter_step_id,
                                                                date_list=dates, field_props=parameter_field_props)
            machine_data = machine_data if machine_data else {}
            _reference_list = []
            for key, value in machine_data.items():
                keys_list = {}
                if not isinstance(value, dict):
                    continue
                for _data, v in value.items():
                    var = _data.split("_")
                    if _data.startswith(tag_tank) or _data.startswith(tag_volume):
                        event_key = "_".join(var[-2:])
                        if event_key not in keys_list:
                            keys_list.update({event_key: {}})
                        if _data.startswith(tag_tank):
                            keys_list[event_key].update({"keys": v})
                        if _data.startswith(tag_volume):
                            keys_list[event_key].update({"values": v})
                for k, v in keys_list.items():
                    if v["keys"] not in return_dict:
                        return_dict.update({v["keys"]: []})
                    if not v["values"] or ("-" in v["values"] and len(v["values"]) == 1):
                        continue
                    if isinstance(v["values"], str) and (
                            all(_chr.isalpha() for _chr in v["values"]) or regex_spl.search(v["values"])):
                        continue
                    try:
                        return_dict[v["keys"]].append(float(v["values"]))
                    except Exception as e:
                        logger.debug(f"Exception occurred while converting values to float {e}")
                        continue
        except Exception as e:
            logger.exception(f'Exception occurred while fetching the machine data {e}')
        return return_dict

    def machine_date_for_multiple_dates(self, step_id, date_list: list, field_props: dict, request_data):
        periodic_dict = {}
        try:
            periodic_data = list(self.periodic_conn.find(query={'step_id': step_id, "date": {'$in': date_list}}))
            trigger_steps = list(self.trigger_step_conn.aggregate(
                pipelines=[{'$match': {'step_id': step_id, "date": {'$in': date_list}}},
                           {'$group': {'_id': None, 'data': {
                               '$push': {'k': {'$ifNull': ['$date', '']}, 'v': {'$ifNull': ['$form_info', '']}}}}},
                           {'$replaceRoot': {'newRoot': {'$arrayToObject': '$data'}}}]))
            trigger_steps = trigger_steps[0] if trigger_steps else {}
            for _data in periodic_data:
                form_df = pd.DataFrame.from_dict(trigger_steps.get(_data.get('date')) or field_props, orient='index')
                form_df = form_df[form_df['time_associated'] == "true"].reset_index().rename(
                    columns={"index": "prop"})
                form_df_time = form_df.copy()
                if _data.get("date") not in periodic_dict:
                    periodic_dict[_data.get('date')] = {}
                if "time_associated" in form_df and _data.get('data'):
                    final_df = self.custom_imp.form_data_df(_data.get('data'), request_data.tz)
                    rounded_df = self.processor.round_off(final_df, "values")
                    current_day = self.processor.merge_with_another_df(form_df_time, rounded_df,
                                                                       merge_on=['tag', 'time'])
                    if "next_day" not in current_day:
                        current_day['next_day'] = ''
                    if "previous_day" not in current_day:
                        current_day['previous_day'] = ''
                    if "default" not in current_day:
                        current_day['default'] = ''
                    field_props = self.custom_imp.merge_relative(current_day)
                    periodic_dict[_data.get('date')] = field_props
                else:
                    periodic_dict[_data.get('date')] = _data.get('manual_entry', {}) if _data.get('manual_entry',
                                                                                                  {}) else {}

        except Exception as e:
            logger.error(f"Exception occurred while fetching the data for multiple dates {e}")
        return periodic_dict

    def received_by(self, user_id):
        return self.user.find_user(user_id).get('username', "")

    @staticmethod
    def last_updated_at():
        return time.time() * 1000

    def save_data_in_master_step(self, request_data, user_id):
        try:
            task_data = self.task_inst_conn.find_by_task_id(request_data.task_id)
            if task_data.master_details.get('auto_save', ""):
                for each in task_data.master_details.get('auto_save', ""):
                    default_obj = getattr(FormRenderingEngine(), each)
                    val = default_obj(user_id)
                    task_data.task_creation_data.update({each: val})
                self.task_inst_conn.update_instance_task(task_data.task_id, data=task_data.dict())
            counter = task_data.master_details.get("task_count", 0)
            updated_dict = {}
            master_task_id = task_data.master_details.get("master_task_id", "")
            master_task_data = self.task_inst_conn.find_by_task_id(master_task_id)
            step_list = []
            if master_task_data:
                prefix_wise_counter = master_task_data.master_details.get('prefix_wise_counter', {})
                prefix = task_data.master_details.get('prefix_key', "")
                prefix_value = prefix_wise_counter.get(prefix, "")
                counter_considered = prefix_value if prefix and prefix_value else counter
                if counter_considered:
                    for key, value in request_data.submitted_data.get('data', {}).items():
                        if prefix:
                            key_string = f"step_data.data.{prefix}_{key}_{counter_considered}"
                        else:
                            key_string = f"step_data.data.{key}_{counter_considered}"
                        updated_dict.update({key_string: value})
                step_list = task_data.master_details.get("master_steps", [])
            if step_list and master_task_id:
                task_data_list = self.task_inst_data.find_data_with_task_id_step_list(master_task_id, step_list)
                stages_list = []
                for each in task_data_list:
                    stages_list.append(each.get("stage_id"))
                if stages_list:
                    self.task_inst_data.update_many_stages(stages_list, updated_dict, )
            return True
        except Exception as e:
            logger.error(f"Exception occurred while saving data in master step {e}")
            raise

    @staticmethod
    def validate_material_info(form_props, request_data: SaveForm, request_obj: Request):
        try:
            input_json = {"project_id": request_data.project_id, "data": {}, "service_type": "subtraction"}
            props_data = get_field_props_by_keys(form_props=form_props, search_keys="material_entry")
            if props_data:
                for key, value in props_data.items():
                    if value == "material_id":
                        if not request_data.submitted_data.get("data", {}).get(key):
                            logger.debug("Material data value doesn't exist")
                            return False, "Material data value doesn't exist"
                        input_json.update({value: request_data.submitted_data.get("data", {}).get(key)})
                        continue
                    if not request_data.submitted_data.get("data", {}).get(key):
                        continue
                    input_json["data"].update({value: request_data.submitted_data.get("data", {}).get(key)})
                api_url = PathToServices.METADATA_SERVICES + 'ilens_config/material/update'
                try:
                    resp = requests.post(url=api_url, cookies=request_obj.cookies,
                                         json=input_json)
                    logger.debug(f"Resp Code:{resp.status_code}")
                    if resp.status_code in CommonStatusCode.SUCCESS_CODES:
                        response = resp.json()
                        logger.debug(f"Response:{response}")
                        if response.get("status") == "failed":
                            return False, response.get("message")
                        return True, response.get("message")
                    return False, "Connection Failure for Updating Material Info"
                except requests.exceptions.ConnectionError as e:
                    logger.exception(e.args)
                    return False, "Connection Failure for Updating Material Info"
        except Exception as e:
            logger.exception(f'Error occurred while validating the material info {e}')
        return True, "Success"

    def stage_filter(self, request_data: SaveForm, property_details, task_instance):
        try:
            if property_details:
                property_key = list(property_details.keys())
                creation_data = task_instance.task_creation_data
                creation_data.update(property_details)
                entered_data = request_data.submitted_data.get("data")
                final_dict = {}
                for i in property_key:
                    values_saved = entered_data.get(i)
                    final_dict.update({i: values_saved})
                self.task_inst.update_task_creation_by_task_id(task_id=request_data.task_id,
                                                               property_dict=final_dict)
        except Exception as e:
            logger.error(f"Exception while saving record {str(e)}")
            raise
