import pandas as pd

from scripts.constants.app_constants import CustomKeys
from scripts.db import PeriodicData, mongo_client
from scripts.logging.logging import logger
from scripts.utils.data_processor import ProcessData


class CustomImplementations:
    def __init__(self, project_id=None):
        self.processor = ProcessData(project_id=project_id)
        self.periodic_conn = PeriodicData(mongo_client, project_id=project_id)

    def form_data_df(self,
                     data_for_day,
                     tz, current_day=False):

        try:
            day_df = pd.DataFrame(columns=['tag', 'time', 'values']) if not current_day else pd.DataFrame(
                columns=['tag', 'time', 'values', 'default'])
            for each_time in data_for_day:
                val = pd.DataFrame(each_time)
                val["ts"] = self.processor.convert_series_format(val["ts"], tz, "%H:%M")
                val = val.reset_index().rename(columns={"index": "tag", "ts": "time"})
                day_df = pd.concat([day_df, val], ignore_index=True)
            return day_df
        except Exception as e:
            logger.error(f"Error in custom implementation in form_data_df, {e}")

    def relative_day(self,
                     relative_data,
                     attribute,
                     form_df,
                     tz, ):
        try:
            relative_day_base = self.form_data_df(relative_data, tz)
            round_relative_day = self.processor.round_off(relative_day_base, "values")
            relative_day_with_attr = self.processor.merge_with_another_df(form_df, round_relative_day,
                                                                          merge_on=['tag', 'time'])
            # Case: No next_day attribute, Handle: empty df returned to prevent overwriting current_day df
            if attribute not in relative_day_with_attr:
                base_df = pd.DataFrame(columns=['tag', 'time', 'values', 'previous_day', 'next_day', 'default'])
                return base_df
            relative_day_df = relative_day_with_attr[relative_day_with_attr[attribute] == "true"]
            return relative_day_df
        except Exception as e:
            logger.error(f"Error in custom implementation of relative_day, {e}")

    def default_day(self,
                    record_in_db,
                    attribute,
                    default_value,
                    form_df,
                    tz, ):
        try:
            relative_day_base = self.form_data_df(record_in_db, tz)
            round_relative_day = self.processor.round_off(relative_day_base, "values")
            relative_day_with_attr = self.processor.merge_with_another_df(form_df, round_relative_day,
                                                                          merge_on=['tag'])
            if attribute not in relative_day_with_attr:
                base_df = pd.DataFrame(columns=['tag', 'time', 'values', 'previous_day', 'next_day', 'default'])
                return base_df
            relative_day_df = relative_day_with_attr[relative_day_with_attr[attribute] == default_value]
            return relative_day_df
        except Exception as e:
            logger.error(f"Error in custom implementation of last_value, {e}")

    @staticmethod
    def merge_relative(*args):
        try:
            merged_df = pd.concat(args).drop_duplicates(['tag', 'time', 'previous_day', 'next_day'],
                                                        keep='last')
            field_props = merged_df.set_index("prop")["values"].to_dict()
            return field_props
        except Exception as e:
            logger.error(f"Error in custom implementation of merge_relative, {e}")

    def month_to_date(self, step_id, to_date, form_df):
        if "calculate_mtd" in form_df:
            mtd_df = form_df[form_df['calculate_mtd'] == "true"]
            ref_keys = mtd_df['mtd_on_key'].to_dict()
            mtd_list = self.periodic_conn.find_mtd(step_id, to_date, ref_keys)
            if not mtd_list:
                return {}
            mtd = mtd_list[0]
            mtd.pop("_id", None)
            if {CustomKeys.ACTUAL_PRODUCTION_MTD, CustomKeys.CAPACITY_FOR_SHIFT_SUMMARY_MTD}.issubset(set(mtd)):
                ope_calculation = mtd[CustomKeys.ACTUAL_PRODUCTION_MTD] / mtd[
                    CustomKeys.CAPACITY_FOR_SHIFT_SUMMARY_MTD] * 100
                if mtd[CustomKeys.CAPACITY_FOR_SHIFT_SUMMARY_MTD] != 0:
                    mtd.update({CustomKeys.OPE_MTD: ope_calculation})
                else:
                    mtd.update({CustomKeys.OPE_MTD: 0})
                mtd.update({CustomKeys.PRODUCTION_LOSS_MTD: 100 - mtd[CustomKeys.OPE_MTD]})
            return mtd
        return {}

    def time_associated(self,
                        form_df,
                        data_req,
                        request_data,
                        next_record,
                        prv_record,
                        latest_record):
        try:
            form_df = form_df[form_df['time_associated'] == "true"].reset_index().rename(
                columns={"index": "prop"})
            form_df_time = form_df.copy()
            final_df = self.form_data_df(data_req, request_data.tz, current_day=True)
            rounded_df = self.processor.round_off(final_df, "values")
            current_day = self.processor.merge_with_another_df(form_df_time, rounded_df, merge_on=['tag', 'time'])
            if "next_day" in current_day:
                current_day = current_day[current_day['next_day'] != "true"]
            if "previous_day" in current_day:
                current_day = current_day[current_day['previous_day'] != "true"]
            if "default" in current_day:
                current_day = current_day[current_day['default'] != "true"]
            # Add field data with attributes as custom properties
            next_day_df = pd.DataFrame(columns=['tag', 'time', 'values', 'next_day'])
            prev_day_df = pd.DataFrame(columns=['tag', 'time', 'values', 'previous_day'])
            default_data = pd.DataFrame(columns=['tag', 'time', 'values', 'default'])
            if all([len(next_record) == 0, len(prv_record) == 0, len(latest_record) == 0]):
                field_props = current_day.set_index("prop")["values"].to_dict()
                return field_props
            if len(next_record) != 0:
                next_day_df = self.relative_day(next_record, "next_day", form_df, request_data.tz)
            if len(prv_record) != 0:
                prev_day_df = self.relative_day(prv_record, "previous_day", form_df, request_data.tz)
            if len(latest_record) != 0:
                default_data = self.default_day(latest_record, "default", "last_value", form_df, request_data.tz)
            field_props = self.merge_relative(default_data, current_day, next_day_df, prev_day_df)
            return field_props
        except Exception as e:
            logger.error(f"Error in custom implementation time_associated, {e}")

    def get_data_dfs(self,
                     form_df,
                     data_req,
                     request_data,
                     next_record):
        try:
            form_df = form_df[form_df['time_associated'] == "true"].reset_index().rename(
                columns={"index": "prop"})
            form_df_time = form_df.copy()
            final_df = self.form_data_df(data_req, request_data.tz)
            rounded_df = self.processor.round_off(final_df, "values")
            current_day = self.processor.merge_with_another_df(form_df_time, rounded_df, merge_on=['tag', 'time'])
            if "next_day" in current_day:
                current_day = current_day[current_day['next_day'] != "true"]
            # Add field data with attributes as next_day
            if next_record:
                next_day_df = self.relative_day(next_record, "next_day", form_df, request_data.tz)
            else:
                next_day_df = pd.DataFrame(columns=["time_associated", "time", "tag", "next_day", "values", "prop"])
            return current_day, next_day_df
        except Exception as e:
            logger.error(f"Error in custom implementation time_associated, {e}")
