import time

from fastapi import Request

from scripts.config.app_configurations import BackFill
from scripts.constants.date_constants import ui_time_format_data
from scripts.core.engine.data_engine import DataEngine
from scripts.core.schemas.stages import TriggerData
from scripts.db import mongo_client, PeriodicData, StepCollection, TriggerStepCollection
from scripts.errors import ImplementationError, RestrictBlanks
from scripts.logging.logging import logger
from scripts.utils.common_utils import CommonUtils


class ComponentManipulation:
    def __init__(self, project_id=None):
        self.step_conn = StepCollection(mongo_client=mongo_client, project_id=project_id)
        self.periodic_conn = PeriodicData(mongo_client, project_id=project_id)
        self.trigger_step_conn = TriggerStepCollection(mongo_client=mongo_client, project_id=project_id)
        self.common_utils = CommonUtils(project_id=project_id)
        self.data_engine = DataEngine(project_id=project_id)
        self.counter = 0

    def parse_component_json(self, abs_component, time_in_str, data_engine_payload, restrict_blanks):
        """real_time: bool, if enabled, values will be fetched from redis otherwise kairos"""
        for itr, component in enumerate(abs_component):
            if "components" in component.keys():
                for eac_itr, each_component in enumerate(component["components"]):
                    if "rows" in each_component.keys() and isinstance(each_component["rows"], list):
                        head_rows = each_component["rows"][:1]
                        tags = [row["components"][0].get("properties", dict()).get("master_tag", "")
                                for row in head_rows[0] if len(row["components"]) > 0][1:]
                        data_engine_payload.update(tag_list=tags)
                        data_from_kairos = self.data_engine.get_tag_values(**data_engine_payload)
                        if not data_from_kairos:
                            data_from_kairos = dict()
                        if data_from_kairos.keys() not in tags:
                            if not any([data_from_kairos.values()]) and restrict_blanks:
                                raise RestrictBlanks
                            _ = {data_from_kairos.update({missing_tag: "-"}) for missing_tag in tags if
                                 missing_tag not in data_from_kairos.keys()}
                        incremented_rows = int(each_component.get("numRows")) + 1
                        cell_list, form_props = self.generate_prop_names(tags, time_in_str, head_rows[0], )
                        each_component["numRows"] = incremented_rows
                        each_component["rows"].append(cell_list)
                        return True, form_props, data_from_kairos
                    elif "columns" in each_component.keys():
                        self.parse_component_json(each_component["columns"], time_in_str, data_engine_payload,
                                                  restrict_blanks)
                    elif "components" in each_component.keys():
                        self.parse_component_json(each_component["components"], time_in_str, data_engine_payload,
                                                  restrict_blanks)
            elif "rows" in component.keys() and isinstance(component["rows"], list):
                head_rows = component["rows"][:1]
                tags = [row["components"][0].get("properties", dict()).get("master_tag", "") for row in head_rows[0]
                        if len(row["components"]) > 0][1:]
                data_engine_payload.update(tag_list=tags)
                data_from_kairos = self.data_engine.get_tag_values(**data_engine_payload)
                if not data_from_kairos:
                    data_from_kairos = dict()
                if data_from_kairos.keys() not in tags:
                    if not any([data_from_kairos.values()]) and restrict_blanks:
                        raise RestrictBlanks
                    _ = {data_from_kairos.update({missing_tag: "-"}) for missing_tag in tags if
                         missing_tag not in data_from_kairos.keys()}
                incremented_rows = int(component.get("numRows")) + 1
                cell_list, form_props = self.generate_prop_names(tags, time_in_str, head_rows[0])
                component["numRows"] = incremented_rows
                component["rows"].append(cell_list)
                return True, form_props, data_from_kairos
        return False, dict(), dict()

    def table_component(self, option, real_time, step_id, data: TriggerData, project_id, request_obj: Request,
                        restrict_blanks):
        try:
            if data.tag_id:
                self.common_utils.publish_data_to_kafka(tag_dict={int(time.time() * 1000): {data.tag_id: data.status}},
                                                        project_id=project_id)
            func = getattr(self, option)
            return func(step_id, data, real_time, project_id, request_obj, restrict_blanks)

        except Exception as e:
            logger.exception(e)
            raise

    def delete_row(self, step_id, data: TriggerData, real_time, project_id, request_obj, restrict_blanks):
        try:
            if not data.trigger_time:
                self.trigger_step_conn.delete_many_triggers(step_id)
                return
            date_in_str = str(self.common_utils.get_time_by_ts(data.trigger_time / 1000, data.tz,
                                                               time_format=ui_time_format_data["yyyy-MM-dd"]))
            self.trigger_step_conn.delete_one_step(step_id, date_in_str)
        except Exception as e:
            logger.exception(e)
            raise

    def add_row(self, step_id, data: TriggerData, real_time, project_id, request_obj, restrict_blanks):
        try:
            if not data.trigger_time:
                data.trigger_time = int(time.time() * 1000)
            else:
                data.from_time = data.trigger_time - BackFill.trigger_step_threshold
                data.to_time = data.trigger_time
            time_in_str = str(
                self.common_utils.get_time_by_ts(data.trigger_time / 1000, data.tz,
                                                 time_format=ui_time_format_data["HH:mm"]))
            date_in_str = str(self.common_utils.get_time_by_ts(data.trigger_time / 1000, data.tz,
                                                               time_format=ui_time_format_data["yyyy-MM-dd"]))
            component_json, from_trigger_step, time_list = self.get_trigger_template(step_id, date_in_str,
                                                                                     time_in_str)
            if not component_json:
                return dict()
            self.counter = component_json.get("event_counter", 0) + 1
            component = component_json.get("field_elements", dict()).get("components")
            data_engine_payload = dict(ignore_stale=real_time,
                                       last_data=real_time,
                                       tz=data.tz,
                                       request_obj=request_obj)
            if data.from_time and data.to_time:
                data_engine_payload.update(from_time=data.from_time, to_time=data.to_time)
            row_added, form_props, data_from_kairos = self.parse_component_json(component, time_in_str,
                                                                                data_engine_payload, restrict_blanks)
            field_props_dot_not = {f'form_info.{k}': v for k, v in form_props.items()}
            if not row_added:
                return dict()
            self.save_trigger_template(date_in_str, component, step_id, field_props_dot_not, time_list)
            if data.manual_entry:
                data_to_kafka = {data.trigger_time: data_from_kairos}
                self.common_utils.publish_data_to_kafka(data_to_kafka, project_id)
            return data_from_kairos
        except Exception as e:
            logger.exception("Failed at trigger add_row", {e})
            raise

    def save_trigger_template(self, t_date, component, parent_step_id, field_props, time_list):
        self.trigger_step_conn.modify_component_json(parent_step_id, t_date, component, field_props, time_list,
                                                     counter=self.counter)

    def get_trigger_template(self, parent_step_id, t_date, time_to_trigger):
        try:
            from_trigger_col = True
            component_json = self.trigger_step_conn.fetch_one_step(step_id=parent_step_id, date=t_date)
            if not component_json:
                component_json = self.step_conn.fetch_one_step(step_id=parent_step_id)
                from_trigger_col = False
            time_list = component_json.get("time_triggered_for", [])
            if time_to_trigger in time_list:
                raise ImplementationError(f"This stage has already been triggered for {time_to_trigger}")
            time_list.append(time_to_trigger)
            return component_json, from_trigger_col, time_list
        except Exception as e:
            logger.error(e)
            raise

    @staticmethod
    def dict_filler(key_name, tag, _time, form_props, manual_entry=False, field_type="number", entity_key=None):
        key_name = key_name.replace('$', "_").replace(":", "_")
        dict_formed = {
            "label": "Label",
            "hideLabel": True,
            "tableView": True,
            "key": key_name,
            "properties": {},
            "type": field_type,
            "input": True
        }
        if tag:
            dict_formed["properties"].update({
                "tag": tag,
                "time": _time,
                "time_associated": "true"
            })
        if manual_entry:
            dict_formed["properties"].update({"manual_entry": "true"})
        if entity_key:
            dict_formed["properties"].update({"entity_key": entity_key})
        form_props.update({key_name: dict_formed["properties"]})
        return dict(components=[dict_formed])

    def generate_prop_names(self, only_tags, time_, head_rows, ):
        form_props = dict()
        mapped_list = list()
        manual_entry_list = [row["components"][0].get("properties", dict()).get(
            "master_manual_entry", "false") == "true" for row in head_rows if
                             len(row["components"]) > 0][1:]
        entity_key_list = [row["components"][0].get("properties", dict()).get("master_entity_key", "") for row in
                           head_rows if len(row["components"]) > 0][1:]
        field_types = [row["components"][0].get("properties", dict()).get("master_field_type", "number") for row in
                       head_rows if len(row["components"]) > 0][1:]
        for itr, x in enumerate(only_tags):
            field = self.dict_filler(key_name=f"{x}_event_{self.counter}",
                                     tag=x,
                                     _time=time_,
                                     form_props=form_props,
                                     manual_entry=manual_entry_list[itr],
                                     field_type=field_types[itr],
                                     entity_key=entity_key_list[itr])
            mapped_list.append(field)

        time_field = {"components": [{
            "label": "Time",
            "hideLabel": True,
            "disabled": True,
            "tableView": True,
            "defaultValue": time_,
            "key": f"event_{self.counter}",
            "type": "time",
            "input": True,
            "inputMask": "99:99"
        }]}
        mapped_list.insert(0, time_field)
        return mapped_list, form_props
