from sqlalchemy import create_engine, inspect
from sqlalchemy.orm import Session

from scripts.config.app_configurations import DBConf
from scripts.db.psql.models.production_losses import DBModel
from scripts.db.psql.query_layer.ope_formula_calculation import QueryLayer as OPEQueryLayer
from scripts.logging.logging import logger, logging_config


class QueryLayer:
    def __init__(self, db: Session):
        self.session: Session = db
        self.table = DBModel
        self.echo = logging_config["level"].upper() == "DEBUG"
        self.create_table(self.table.__tablename__)

    def create_table(self, table_name):
        try:
            engine = create_engine(DBConf.ASSISTANT_DB_URI, echo=self.echo)
            if not inspect(engine).has_table(table_name):
                orm_table = self.table
                orm_table.__table__.create(bind=engine, checkfirst=True)
        except Exception as e:
            logger.error(f"Error occurred during start-up: {e}", exc_info=True)

    @property
    def column_id(self):
        return "id"

    @property
    def column_date(self):
        return "date"

    @property
    def column_step_id(self):
        return "step_id"

    @property
    def column_line(self):
        return "line"

    @property
    def column_equipment(self):
        return "equipment"

    @property
    def column_loss_category(self):
        return "loss_category"

    @property
    def column_loss_reason(self):
        return "loss_reason"

    @property
    def column_loss_reason_comment(self):
        return "loss_reason_comment"

    @property
    def column_loss_in_minutes(self):
        return "loss_in_minutes"

    def add_to_table(self, input_tuple, step_id, date):
        list_of_recs = list()
        try:
            dg, unaccounted_loss, booked_loss, refer_step = input_tuple
            OPEQueryLayer(self.session).update_losses(unaccounted_loss, booked_loss, date, refer_step)
            for each in dg:
                each["shift"] = each["shifts"]
                each.pop("shifts")
                table_obj = self.table(**each, step_id=step_id, date=date)
                list_of_recs.append(table_obj)
            return list_of_recs
        except TypeError:
            raise
        except Exception as e:
            logger.exception(f"Exception occurred while adding to postgres table {e}")

    def insert_data(self, object_models_list, step_id, date):
        try:
            self.delete_data(step_id, date)
            mappings = self.add_to_table(object_models_list, step_id, date)
            self.session.bulk_save_objects(mappings)
            self.session.commit()
            return True
        except TypeError:
            raise
        except Exception as e:
            logger.exception(e)
            raise

    def delete_data(self, step_id, date):
        try:
            self.session.query(self.table) \
                .filter(self.table.step_id == step_id, self.table.date == date) \
                .delete()
            self.session.commit()
            return True
        except Exception as e:
            logger.error(f"Failed to delete existing data, {e}")
            raise
