from typing import Optional, Dict

from scripts.constants.app_constants import DatabaseNames, CollectionNames
from scripts.db.mongo.schema import MongoBaseSchema
from scripts.utils.mongo_util import MongoCollectionBaseClass


class UserRoleSchema(MongoBaseSchema):
    type: Optional[str]
    user_role_name: Optional[str]
    user_role_description: Optional[str]
    user_role_id: Optional[str]
    project_id: Optional[str]
    user_role_permissions: Optional[Dict]
    access_levels: Optional[Dict]
    default: Optional[bool]
    client_id: Optional[str]
    product_encrypted: Optional[bool]
    permission_status: Optional[bool]


class UserRoleCollectionKeys:
    KEY_USER_ROLE_ID = "user_role_id"
    KEY_PROJECT_ID = "project_id"


class UserRole(MongoCollectionBaseClass):
    def __init__(self, mongo_client):
        super().__init__(mongo_client, database=DatabaseNames.ilens_configuration,
                         collection=CollectionNames.user_role)

    @property
    def key_user_role_id(self):
        return UserRoleCollectionKeys.KEY_USER_ROLE_ID

    @property
    def key_project_id(self):
        return UserRoleCollectionKeys.KEY_PROJECT_ID

    def find_roles(self):
        access_groups = self.find({})
        if access_groups:
            return access_groups
        return list()

    def find_role_by_project(self, project_id):
        query = {self.key_project_id: project_id}
        access_groups = self.find_one(query)
        if access_groups:
            return access_groups
        return list()

    def find_role_by_param(self, **query):
        access_groups = self.find_one(query)
        if access_groups:
            return access_groups
        return dict()

    def find_roles_by_list(self, user_role_id_list, project_id):
        query = {"$or": [{self.key_user_role_id: {"$in": user_role_id_list}}]}
        if self.key_project_id is not None:
            query["$or"].append({self.key_project_id: project_id})
        access_groups = self.find(query)
        if access_groups:
            return access_groups
        return list()

    def find_user_role_by_id(self, user_role_id, filter_dict=None):
        user = self.find_one(query={self.key_user_role_id: user_role_id}, filter_dict=filter_dict)
        return user

    def update_user_role(self, _id, data):
        query = {self.key_user_role_id: _id}
        self.update_one(query=query, data=data)

    def delete_user_role(self, _id):
        if _id:
            query = {self.key_user_role_id: _id}
            self.delete_one(query=query)
        else:
            return False

    def get_data_by_aggregate(self, query_json: list):
        response = list(self.aggregate(query_json))
        return response
