from typing import Optional, Dict, List

from scripts.constants.app_constants import DatabaseNames, CollectionNames
from scripts.constants.db_keys import WorkflowKeys
from scripts.db.mongo.schema import MongoBaseSchema
from scripts.utils.mongo_util import MongoCollectionBaseClass


class WorkflowSchema(MongoBaseSchema):
    """
    This is the Schema for the Mongo DB Collection.
    All datastore and general responses will be following the schema.
    """
    workflow_id: Optional[str]
    workflow_name: Optional[str]
    tags: Optional[list] = []
    description: Optional[str]
    meta: Optional[Dict]
    steps: Optional[List] = []
    roles: Optional[List] = []
    permissions: Optional[List] = []
    project_id: Optional[str]
    is_deleted: bool = False
    workflow_version: Optional[str]
    validation: Optional[Dict] = {}


class Workflow(MongoCollectionBaseClass):
    def __init__(self, mongo_client, project_id=None):
        super().__init__(mongo_client, database=DatabaseNames.ilens_assistant,
                         collection=CollectionNames.workflows)
        self.project_id = project_id

    @property
    def key_workflow_id(self):
        return WorkflowKeys.KEY_WORKFLOW_ID

    @property
    def key_project_id(self):
        return WorkflowKeys.KEY_PROJECT_ID

    @property
    def key_workflow_version(self):
        return WorkflowKeys.KEY_WORKFLOW_VERSION

    @property
    def key_workflow_name(self):
        return WorkflowKeys.KEY_WORKFLOW_NAME

    def find_by_id(self, workflow_id: str, workflow_version):
        query = {self.key_workflow_id: workflow_id, self.key_workflow_version: workflow_version}
        record = self.find_one(query)
        if not record:
            return WorkflowSchema(**dict())
        return WorkflowSchema(**record)

    def find_name_by_id(self, workflow_id: str):
        query = {self.key_workflow_id: workflow_id}
        filter_dict = {self.key_workflow_name: 1, "_id": 0}
        record = self.find_one(query, filter_dict)
        if not record:
            return None
        return record[self.key_workflow_name]

    def fetch_workflow_record(self, **query):
        record = self.find_one(query)
        if not record:
            return None
        return record

    def add_new_workflow(self, data: dict):
        data = WorkflowSchema(**data).dict()
        self.insert_one(data)
        return WorkflowSchema(**data).workflow_id

    def update_workflow(self, workflow_id, data: dict, upsert=False):
        query = {self.key_workflow_id: workflow_id}
        return self.update_one(data=data, query=query, upsert=upsert)

    def soft_delete_workflow(self, workflow_id, workflow_version):
        json_update = {"is_deleted": True}
        query = {self.key_workflow_id: workflow_id, self.key_workflow_version: workflow_version}
        return self.update_one(data=json_update, query=query)

    def find_all_workflows(self, project_id: Optional[str]):
        query = {self.key_project_id: project_id} if project_id else {}
        records = self.find(query)
        if not records:
            return list()
        return records

    def find_all_undeleted_workflows(self, project_id: Optional[str]):
        query = {self.key_project_id: project_id, "is_deleted": False} if project_id else {"is_deleted": False}
        records = self.find(query)
        if not records:
            return list()
        return records

    def get_data_by_aggregate(self, query_json: list):
        response = list(self.aggregate(query_json))
        return response
