from typing import Optional, Dict

from scripts.constants.app_constants import DatabaseNames, CollectionNames
from scripts.constants.db_keys import TriggerKeys, WorkflowKeys
from scripts.db.mongo.schema import MongoBaseSchema
from scripts.utils.mongo_util import MongoCollectionBaseClass


class TriggerWorkflowSchema(MongoBaseSchema):
    """
    This is the Schema for the Mongo DB Collection.
    All datastore and general responses will be following the schema.
    """
    workflow_id: Optional[str]
    workflow_version: Optional[int]
    trigger_id: Optional[str]
    trigger_type: Optional[str]
    actions: Optional[list] = []
    trigger_meta: Optional[Dict] = {}


class Trigger(MongoCollectionBaseClass):
    def __init__(self, mongo_client, project_id=None):
        super().__init__(mongo_client, database=DatabaseNames.ilens_assistant,
                         collection=CollectionNames.triggers)
        self.project_id = project_id

    @property
    def key_trigger_id(self):
        return TriggerKeys.KEY_TRIGGER_ID

    @property
    def key_workflow_version(self):
        return WorkflowKeys.KEY_WORKFLOW_VERSION

    @property
    def key_workflow_id(self):
        return WorkflowKeys.KEY_WORKFLOW_ID

    def update_trigger(self, trigger_id, data: dict, upsert=False):
        query = {self.key_trigger_id: trigger_id}
        return self.update_one(data=data, query=query, upsert=upsert)

    def soft_delete_trigger(self, trigger_id):
        json_update = {"is_deleted": True}
        query = {self.key_trigger_id: trigger_id}
        return self.update_one(data=json_update, query=query)

    def find_by_id(self, workflow_id: str, workflow_version: int):
        query = {self.key_workflow_id: workflow_id, self.key_workflow_version: workflow_version}
        record = self.find_one(query)
        if not record:
            return TriggerWorkflowSchema(**dict())
        return TriggerWorkflowSchema(**record)

    def fetch_by_id(self, **query):
        if 'role' in query:
            query["trigger_meta.role"] = query['role']
            query.pop('role')
        if 'on_click' in query:
            query["trigger_meta.on_click"] = query['on_click']
            query.pop('on_click')
        record = self.find_one(query)
        if not record:
            return TriggerWorkflowSchema(**dict())
        return TriggerWorkflowSchema(**record)

    def find_all_triggers(self, workflow_id: Optional[str], workflow_version: Optional[str]):
        query = {self.key_workflow_id: workflow_id, self.key_workflow_version: workflow_version,
                 "is_deleted": False} if workflow_id and workflow_version else {}
        records = self.find(query=query)
        if not records:
            return list()
        return list(records)
