from typing import Dict, List, Optional

from scripts.constants.app_constants import DatabaseNames, CollectionNames
from scripts.constants.db_keys import StepRecordKeys
from scripts.db.mongo.schema import MongoBaseSchema
from scripts.utils.mongo_util import MongoCollectionBaseClass


class StepSchema(MongoBaseSchema):
    """
    This is the Schema for the Mongo DB Collection.
    All datastore and general responses will be following the schema.
    """
    step_id: str
    meta: Dict
    field_elements: Dict
    form_factor: List
    description: str
    project_id: str
    step_sub_category: Optional[str]
    step_name: str
    step_category: str
    display_title: str
    event_counter: Optional[int]
    replicate_type: Optional[str]
    auto_save: Optional[str]
    display_properties: Optional[List] = []


class StepCollection(MongoCollectionBaseClass):
    def __init__(self, mongo_client, project_id=None):
        super().__init__(mongo_client, database=DatabaseNames.ilens_assistant,
                         collection=CollectionNames.steps)
        self.project_id = project_id

    @property
    def key_project_id(self):
        return StepRecordKeys.KEY_PROJECT_ID

    @property
    def key_step_id(self):
        return StepRecordKeys.KEY_STEP_ID

    @property
    def key_step_name(self):
        return StepRecordKeys.KEY_STEP_NAME

    def insert_one_step(self, data):
        """
        The following function will insert one tag in the
        tags collections
        :param self:
        :param data:
        :return:
        """
        return self.insert_one(data)

    def update_one_step(self, data, upsert=False, **query):
        """
        The following function will update one step in
        steps collection based on the given query
        :param data:
        :param upsert:
        :param query:
        :return:
        """
        return self.update_one(data=data, upsert=upsert, query=query)

    def delete_one_step(self, **query):
        """
        The following function will delete one tag in
        tags collection based on the given query
        :param query:
        :return:
        """
        return self.delete_one(query=query)

    def find_many_by_project_id(self, project_id):
        """
        The following function will give one process for a given set of
        search parameters as keyword arguments
        :param project_id:
        :return:
        """
        query = {self.key_project_id: project_id}
        many_process = self.find(query=query)
        if not many_process:
            return list()
        return list(many_process)

    def fetch_one_step(self, filter_dict=None, **query):
        one_step = self.find_one(filter_dict=filter_dict, query=query)
        if not one_step:
            return None
        return one_step

    def fetch_step_name(self, step_id):
        query = {self.key_step_id: step_id}
        one_step = self.find_one(filter_dict={"step_name": 1, "_id": 0}, query=query)
        return one_step["step_name"]

    def map_positions_for_templates(self, template_list):
        query = {self.key_step_id: {"$in": template_list}}
        templates = self.find(query, {"step_id": 1, "_id": 0, "menu_placement": 1})
        if not templates:
            return list()
        return [dict(menu_placement=x["menu_placement"], step_id=x["step_id"]) for x in templates]

    def get_step_map(self, steps):
        query = {self.key_step_id: {"$in": steps}}
        steps_data = self.find(query, {"step_id": 1, "menu_placement": 1, "_id": 0})
        if not steps_data:
            return list()
        step_map = {}
        for step in steps_data:
            step_map.update({step.get("step_id"): step.get("menu_placement")})
        return step_map

    def find_many(self, step_list):
        query = {"step_id": {"$in": step_list}}
        steps = self.find(query)
        if not steps:
            return dict()
        return {step["step_id"]: step["field_elements"] for step in steps}

    def get_data_by_aggregate(self, query_json: list):
        return list(self.aggregate(query_json))

    def modify_component_json(self, step_id, component_json, counter=None):
        query = {"step_id": step_id}
        data = {"field_elements": {"components": component_json}}
        if counter:
            data.update({"event_counter": counter})
        self.update_one(query, data, upsert=False)

    def find_by_id(self, step_id: str):
        query = {"step_id": step_id}
        record = self.find_one(query)
        if not record:
            return StepSchema(**dict())
        return StepSchema(**record)
