from datetime import date
from typing import Optional, List, Dict

from scripts.constants.app_constants import DatabaseNames, CollectionNames
from scripts.constants.db_keys import PeriodicDataKeys
from scripts.db.mongo.ilens_assistant.aggregations.periodic_data import PeriodicDataAgg
from scripts.db.mongo.schema import MongoBaseSchema
from scripts.utils.mongo_util import MongoCollectionBaseClass


class PeriodicDataSchema(MongoBaseSchema):
    """
    This is the Schema for the Mongo DB Collection.
    All datastore and general responses will be following the schema.
    """
    step_id: Optional[str]
    date: Optional[str]
    data: Optional[List] = []
    manual_data: Optional[Dict] = dict()


class PeriodicData(MongoCollectionBaseClass):
    def __init__(self, mongo_client, project_id=None):
        super().__init__(mongo_client, database=DatabaseNames.ilens_assistant,
                         collection=CollectionNames.periodic_data)
        self.project_id = project_id

    @property
    def key_step_id(self):
        return PeriodicDataKeys.KEY_STEP_ID

    @property
    def key_date(self):
        return PeriodicDataKeys.KEY_DATE

    @property
    def key_data(self):
        return PeriodicDataKeys.KEY_DATA

    @property
    def key_manual_data(self):
        return PeriodicDataKeys.KEY_MANUAL_DATA

    def find_by_id(self, step_id: str):
        query = {self.key_step_id: step_id}
        record = self.find_one(query)
        if not record:
            return None
        return PeriodicDataSchema(**record)

    def find_by_date_and_step(self, _date: str, step_id: str):
        query = {self.key_date: _date, self.key_step_id: step_id}
        record = self.find_one(query)
        if not record:
            return PeriodicDataSchema()
        return PeriodicDataSchema(**record)

    def find_by_latest_data(self, step_id, current_date, default_tags):
        granular_tag_query = {f'data.values.{x}': {'$exists': 1} for x in default_tags}
        query = PeriodicDataAgg.get_latest_value(step_id, current_date, granular_tag_query)
        records = self.aggregate(query)
        tag_dict = {}
        # TODO: Can this go on aggregation pipeline?
        for record in records:
            values = record.get("data", {}).get("values", {})
            for key, val in values.items():
                if key not in tag_dict and key in default_tags:
                    tag_dict.update({key: val})
        return dict(ts=0, values=tag_dict)

    def find_mtd(self, step_id, to_date, mtd_keys):
        mtd_keys_match = {x: {"$sum": f"$manual_data.{y}"} for x, y in mtd_keys.items()}
        query = PeriodicDataAgg.month_to_date(step_id, to_date, mtd_keys_match)
        records = self.aggregate(query)
        return list(records)

    def find_by_date_and_multi_step(self, _date: str, step_id_list: List):
        query = {self.key_date: _date, self.key_step_id: {"$in": step_id_list}}
        record = self.find(query)
        if not record:
            return list()
        return record

    def insert_date(self, _date: date, step_id: str, data):
        json_data = {self.key_date: _date, self.key_step_id: step_id, self.key_data: data}
        self.insert_one(json_data)
        return True

    def update_data_with_date(self, _date: str, step_id: str, data):
        query = {self.key_date: _date, self.key_step_id: step_id}
        self.update_to_set(query, self.key_data, data, upsert=True)
        return True

    def update_data_with_date_periodic(self, _date: str, step_id: str, manual_data):
        query = {self.key_date: _date, self.key_step_id: step_id}
        self.update_one(query, data=manual_data, upsert=True)
        return True

    def save_and_update_periodic_data(self, _date: str, step_id: str, periodic_data):
        query = {self.key_date: _date, self.key_step_id: step_id}
        existing_record = self.find_by_date_and_step(_date, step_id)
        ts_list = list()
        if existing_record.data:
            for each in existing_record.data:
                if "values" in each and each["ts"] == periodic_data.get("ts"):
                    values_dict = each.get("values", {})
                    values_dict.update(periodic_data.get("values", {}))
                    ts_list.append(each["ts"])
            if periodic_data.get("ts") not in ts_list:
                existing_record.data.append({"ts": periodic_data.get("ts"), "values": periodic_data.get("values")})
            periodic_data = existing_record.data
        else:
            periodic_data = [dict(ts=periodic_data.get("ts"), values=periodic_data.get("values"))]
        data = {self.key_date: _date, self.key_step_id: step_id, self.key_data: periodic_data}
        self.update_one(query, data, upsert=True)
        return True

    def save_and_update_data(self, _date: str, step_id: str, data):
        manual_data, periodic_data = data.get("manual_data"), data.get("data")
        query = {self.key_date: _date, self.key_step_id: step_id}
        existing_record = self.find_by_date_and_step(_date, step_id)
        ts_list = list()
        if existing_record.data:
            for each in existing_record.data:
                if "values" in each and each["ts"] in periodic_data:
                    ts = each["ts"]
                    ts_list.append(ts)
                    each["values"].update(periodic_data[ts])
            for ts, values in periodic_data.items():
                if ts not in ts_list:
                    existing_record.data.append({"ts": ts, "values": values})
            periodic_data = existing_record.data
        else:
            periodic_data = [dict(ts=x, values=y) for x, y in periodic_data.items()]

        if not manual_data and not periodic_data:
            return True
        if manual_data and periodic_data:
            existing_record.manual_data.update(manual_data)
            data = {self.key_date: _date, self.key_step_id: step_id, self.key_data: periodic_data,
                    self.key_manual_data: existing_record.manual_data}
        elif periodic_data:
            data = {self.key_date: _date, self.key_step_id: step_id, self.key_data: periodic_data}
        else:
            existing_record.manual_data.update(manual_data)
            data = {self.key_date: _date, self.key_step_id: step_id, self.key_manual_data: existing_record.manual_data}
        self.update_one(query, data, upsert=True)
        return True

    def find_data_with_date(self, _date_query: dict, step_id: str, sort_json={}):
        query = {self.key_date: _date_query, self.key_step_id: step_id}
        record = self.find(query, sort=list(sort_json.items()))
        if not record:
            return list()
        return record
