import traceback
from typing import Optional, Any

from fastapi import APIRouter, Depends, Request
from sqlalchemy.orm import Session
from starlette.background import BackgroundTasks

from scripts.constants.api import FormEndPoints
from scripts.constants.app_constants import AuditingKeys
from scripts.core.engine.form_renderer import FormRenderingEngine
from scripts.core.handlers.stage_handler import StageHandler
from scripts.core.schemas.response_models import DefaultResponse, DefaultFailureResponse
from scripts.core.schemas.stages import StagesList, GetKDataRequest, TriggerData, TriggerReferenceData, \
    MarkTaskCompleteRequest, SendNotification, CopyPropertyValues
from scripts.db.psql.databases import get_db
from scripts.errors import StepsNotConfigured, ImplementationError, RestrictBlanks
from scripts.logging.logging import logger
from scripts.utils.security_utils.decorators import CookieAuthentication
from scripts.utils.security_utils.project_decorator import MetaInfoCookie, MetaInfoSchema

stage_router = APIRouter(tags=["Stage services"], prefix=FormEndPoints.api_stage)
auth = CookieAuthentication()
get_cookies = MetaInfoCookie()


@stage_router.post(FormEndPoints.api_list)
async def get_stages_list(request_data: StagesList, nav_type: Optional[str] = "", user_id=Depends(auth),
                          meta: MetaInfoSchema = Depends(get_cookies)):
    try:
        stage_handler = StageHandler(project_id=meta.project_id)
        response = stage_handler.get_stages_list(request_data, user_id, nav_type)
        return DefaultResponse(status="success", message="success", data=response).dict()
    except StepsNotConfigured:
        return DefaultFailureResponse(message="Steps are not configured for the selected task")
    except PermissionError as e:
        return DefaultFailureResponse(error="Permission Error",
                                      message=f"Permission Error, possible causes: <br/> "
                                              f"{e.args[0]}")
    except Exception as e:
        tb = traceback.format_exc()
        logger.exception(e)
        logger.exception(tb)
        return DefaultFailureResponse(error=e.args, message=e.args).dict()


@stage_router.post(FormEndPoints.api_add_periodic_data)
async def add_periodic_stage_data(request_data: GetKDataRequest, meta: MetaInfoSchema = Depends(get_cookies)):
    try:
        stage_handler = StageHandler(project_id=meta.project_id)
        response = stage_handler.add_periodic_stage_data(request_data)
        return DefaultResponse(status="success", message="success", data=response).dict()
    except StepsNotConfigured:
        return DefaultFailureResponse(message="Steps are not configured for the selected task")
    except Exception as e:
        tb = traceback.format_exc()
        logger.exception(e)
        logger.exception(tb)
        return DefaultFailureResponse(error=e.args, message=e.args).dict()


@stage_router.get(FormEndPoints.api_get_tags)
async def get_tags(step_id: str, missing: bool = False, meta: MetaInfoSchema = Depends(get_cookies)):
    try:
        stage_handler = StageHandler(project_id=meta.project_id)
        response = stage_handler.get_tags(step_id, missing)
        return DefaultResponse(status="success", message="success", data=response).dict()
    except Exception as e:
        tb = traceback.format_exc()
        logger.exception(e)
        logger.exception(tb)
        return DefaultFailureResponse(error=e.args, message=e.args).dict()


@stage_router.get(FormEndPoints.api_get_time_list)
async def get_time_list(step_id: str, meta: MetaInfoSchema = Depends(get_cookies)):
    try:
        stage_handler = StageHandler(project_id=meta.project_id)
        response = stage_handler.get_time_list(step_id)
        return DefaultResponse(status="success", message="success", data=response).dict()
    except Exception as e:
        tb = traceback.format_exc()
        logger.exception(e)
        logger.exception(tb)
        return DefaultFailureResponse(error=e.args, message=e.args).dict()


@stage_router.get(FormEndPoints.api_timewise_tags)
async def get_tag_and_time_list(step_id: str, meta: MetaInfoSchema = Depends(get_cookies)):
    try:
        stage_handler = StageHandler(project_id=meta.project_id)
        response = await stage_handler.get_tag_and_time_list(step_id)
        return DefaultResponse(status="success", message="success", data=response).dict()
    except Exception as e:
        tb = traceback.format_exc()
        logger.exception(e)
        logger.exception(tb)
        return DefaultFailureResponse(error=e.args, message=e.args).dict()


@stage_router.get(FormEndPoints.api_trigger)
async def trigger_data(step_id: str,
                       tz: str,
                       request: Request,
                       trigger_time: Optional[int] = None,
                       option: str = "add_row",
                       trigger_method: Optional[str] = AuditingKeys.machine,
                       status: Optional[int] = None,
                       tag_id: Optional[str] = None,
                       real_time: Optional[bool] = True,
                       restrict_blanks: Optional[bool] = False,
                       meta: MetaInfoSchema = Depends(get_cookies)):
    try:
        stage_handler = StageHandler(project_id=meta.project_id)
        trigger_model = TriggerData(step_id=step_id, tz=tz, trigger_method=trigger_method, status=status, tag_id=tag_id)
        if trigger_time:
            trigger_model.trigger_time = trigger_time
        response = await stage_handler.add_triggered_data(option, trigger_model, real_time, restrict_blanks,
                                                          project_id=meta.project_id,
                                                          request_obj=request)
        return DefaultResponse(status="success", message="Trigger operation successful", data=response).dict()
    except RestrictBlanks:
        return DefaultResponse(
            status="success",
            message="Triggered successfully. Row not populated since Restrict blanks was enabled").dict()
    except ModuleNotFoundError:
        return DefaultResponse(
            message="Error occurred while connecting to Periodic Data Engine. Please Contact Admin").dict()
    except ImplementationError as message:
        return DefaultResponse(message=message.args[0]).dict()
    except Exception as e:
        tb = traceback.format_exc()
        logger.exception(e)
        logger.exception(tb)
        return DefaultFailureResponse(error=e.args, message=e.args).dict()


@stage_router.get(FormEndPoints.api_trigger_task_completion)
async def trigger_task_completion(factor: str, meta: MetaInfoSchema = Depends(get_cookies)):
    try:
        stage_handler = StageHandler(project_id=meta.project_id)
        response = await stage_handler.trigger_task_completion(factor)
        return DefaultResponse(status="success", message="success", data=response).dict()
    except KeyError:
        return DefaultResponse(
            message="Key Error encountered while updating triggered tasks").dict()
    except Exception as e:
        tb = traceback.format_exc()
        logger.exception(e)
        logger.exception(tb)
        return DefaultFailureResponse(error=e.args, message=e.args).dict()


@stage_router.put(FormEndPoints.api_mark_task_complete)
async def mark_task_complete(request_data: MarkTaskCompleteRequest,
                             meta: MetaInfoSchema = Depends(get_cookies)):
    try:
        stage_handler = StageHandler(project_id=meta.project_id)
        response = await stage_handler.mark_task_complete(request_data)
        return DefaultResponse(status="success", message="success", data=response).dict()
    except KeyError:
        return DefaultResponse(
            message="Key Error encountered while marking task complete").dict()
    except Exception as e:
        tb = traceback.format_exc()
        logger.exception(e)
        logger.exception(tb)
        return DefaultFailureResponse(error=e.args, message=e.args).dict()


@stage_router.get(FormEndPoints.api_list_periodic_steps)
async def get_periodic_steps(skip: int, limit: int = 5, meta: MetaInfoSchema = Depends(get_cookies)):
    try:
        stage_handler = StageHandler(project_id=meta.project_id)
        response = await stage_handler.get_periodic_steps(skip, limit)
        return DefaultResponse(status="success", message="success", data=response).dict()
    except Exception as e:
        tb = traceback.format_exc()
        logger.exception(tb)
        return DefaultFailureResponse(error=e.args, message=e.args).dict()


@stage_router.get(FormEndPoints.api_reference)
async def trigger_reference_data(
        user_id: str, stage_id: str, task_id: str, tz: str, project_id: str,
        current_status: str, date: Optional[str] = None, entity_name: Optional[str] = None,
        entity_key: Optional[str] = None, property_value: Optional[Any] = None,
        row_unique_key: Optional[str] = None, field_type: Optional[str] = "textfield",
        entity_search: Optional[str] = "true", meta: MetaInfoSchema = Depends(get_cookies)
):
    try:
        form_render = FormRenderingEngine(project_id=meta.project_id)
        if not row_unique_key or not property_value or not entity_key or (
                not entity_name and entity_search.lower() != "false"):
            return DefaultResponse(status="success", message="success").dict()
        request_data = TriggerReferenceData(user_id=user_id, stage_id=stage_id, task_id=task_id, tz=tz,
                                            property_value=property_value, entity_key=entity_key,
                                            project_id=project_id,
                                            row_unique_key=row_unique_key, field_type=field_type,
                                            current_status=current_status)
        if date:
            request_data.triggers = dict(date=date)
        response = await form_render.form_fill_with_reference_data(input_data=request_data,
                                                                   entity_search=entity_search, entity_name=entity_name)
        return DefaultResponse(status="success", message="success", data=dict(submitted_data=response)).dict()
    except ModuleNotFoundError:
        return DefaultResponse(
            message="Error occurred while connecting to Reference Data Engine. Please Contact Admin").dict()
    except ImplementationError as message:
        return DefaultResponse(message=message.args[0]).dict()
    except Exception as e:
        tb = traceback.format_exc()
        logger.exception(e)
        logger.exception(tb)
        return DefaultFailureResponse(error=e.args, message=e.args).dict()


@stage_router.post(FormEndPoints.api_send_notification)
async def send_notification(request_data: SendNotification, meta: MetaInfoSchema = Depends(get_cookies)):
    try:
        stage_handler = StageHandler(project_id=meta.project_id)
        response = await stage_handler.send_notification_for_roles(request_data)
        return DefaultResponse(status="success", message="success", data=response).dict()
    except Exception as e:
        tb = traceback.format_exc()
        logger.exception(tb)
        return DefaultFailureResponse(error=e.args, message=e.args).dict()


@stage_router.post(FormEndPoints.api_copy_property_values)
async def copy_property_values(request_data: CopyPropertyValues,
                               bg_task: BackgroundTasks,
                               request: Request,
                               db: Session = Depends(get_db),
                               meta: MetaInfoSchema = Depends(get_cookies),
                               user_id=Depends(auth),
                               ):
    try:
        stage_handler = StageHandler(project_id=meta.project_id)
        response = await stage_handler.copy_property_values(request_data, user_id, bg_task, request, db)
        return DefaultResponse(status="success", message="success", data=response).dict()
    except Exception as e:
        tb = traceback.format_exc()
        logger.exception(tb)
        return DefaultFailureResponse(error=e.args, message=e.args).dict()
