"""Services below are only concerned with CRUD operations associated to:
Workflow Stages
These are stages created in each instance
"""
import traceback
from typing import Optional

from fastapi import APIRouter, Depends, BackgroundTasks, Request
from sqlalchemy.orm import Session

from scripts.constants.api import FormEndPoints
from scripts.core.handlers.form_handler import FormHandler
from scripts.core.schemas.forms import SaveForm
from scripts.core.schemas.response_models import DefaultResponse, DefaultFailureResponse
from scripts.db.psql.databases import get_db
from scripts.errors import ILensPermissionError, RequiredFieldMissing, InternalError, QuantityGreaterThanException
from scripts.logging.logging import logger
from scripts.utils.security_utils.decorators import CookieAuthentication
from scripts.utils.security_utils.project_decorator import MetaInfoCookie, MetaInfoSchema

form_router = APIRouter(tags=["Form services"], prefix=FormEndPoints.api_render)
auth = CookieAuthentication()
get_cookies = MetaInfoCookie()


@form_router.post(FormEndPoints.api_form)
async def form_rendering_service(request_data: SaveForm,
                                 request: Request,
                                 bg_task: BackgroundTasks,
                                 save: Optional[bool] = False,
                                 submit: Optional[bool] = False,
                                 mobile: Optional[bool] = False,
                                 user_id=Depends(auth),
                                 db: Session = Depends(get_db),
                                 meta: MetaInfoSchema = Depends(get_cookies)
                                 ):
    try:
        form_handler = FormHandler(project_id=meta.project_id)
        if not user_id:
            raise ILensPermissionError("Denied: Unauthorized User")
        request_data.user_id = user_id
        message, response = await form_handler.form_renderer(request_data, user_id, request, save, submit, mobile, db,
                                                             bg_task)
        if message:
            return DefaultResponse(status="success", message=message, data=response).dict()
        return DefaultResponse(status="success", data=response).dict()
    except QuantityGreaterThanException as e:
        return DefaultFailureResponse(message=e.args[0]).dict()
    except InternalError as e:
        return DefaultResponse(message=e.args[0], status="warning")
    except RequiredFieldMissing as e:
        logger.info(f"Required fields have not been filled by user")
        return DefaultResponse(message=e.args[0])
    except ILensPermissionError as e:
        return DefaultResponse(message=e.args[0])
    except Exception as e:
        tb = traceback.format_exc()
        logger.exception(e)
        logger.exception(tb)
        return DefaultFailureResponse(error=e.args, message="Error encountered in rendering or saving form")
