import os
import re
from copy import deepcopy

import openpyxl
import pandas as pd
import requests
from fastapi import Request

from scripts.config.app_configurations import PathToStorage, PathToServices
from scripts.constants.api import StageDataEndPoints
from scripts.constants.app_constants import CommonStatusCode
from scripts.constants.date_constants import date_time_with_hour
from scripts.constants.template_constants import TemplateStorage
from scripts.core.schemas.stages_data import TemplateListRequest, TemplateListResponse, \
    FetchTemplate, TemplateTableOptions, TemplateKeyValuePairs, UploadedFileList, UploadedFileListResponse, \
    DeleteDataFile, DeleteTemplate
from scripts.db import mongo_client, User
from scripts.db.mongo.ilens_assistant.aggregations.logbook import LogbookInfoAggregate
from scripts.db.mongo.ilens_assistant.aggregations.step_data_files import StepDataFilesAggregate
from scripts.db.mongo.ilens_assistant.aggregations.step_templates import TemplateInfoAggregate
from scripts.db.mongo.ilens_assistant.collections.logbook import LogbookInfo
from scripts.db.mongo.ilens_assistant.collections.step_data_files import StepDataFiles
from scripts.db.mongo.ilens_assistant.collections.step_templates import StepTemplates
from scripts.db.mongo.ilens_configuration.aggregations.users import UsersAggregate
from scripts.db.mongo.ilens_configuration.collections.constants import Constants
from scripts.errors import DuplicateTemplateNameError, BulkUploadError, ColumnsMisMatch, InvalidValueFound
from scripts.logging.logging import logger
from scripts.utils.common_utils import CommonUtils
from scripts.utils.ilens_publish_data import DataPush


class StagesData:
    def __init__(self, project_id=None):
        self.template_aggregate = TemplateInfoAggregate()
        self.step_template = StepTemplates(mongo_client=mongo_client, project_id=project_id)
        self.logbook_conn = LogbookInfo(mongo_client=mongo_client, project_id=project_id)
        self.logbook_aggregate = LogbookInfoAggregate()
        self.common_utils = CommonUtils(project_id=project_id)
        self.user_aggregate = UsersAggregate()
        self.user = User(mongo_client=mongo_client)
        self.const_conn = Constants(mongo_client=mongo_client)
        self.step_data_files_conn = StepDataFiles(mongo_client=mongo_client, project_id=project_id)
        self.step_data_files_agg = StepDataFilesAggregate()
        self.kafka_data_push = DataPush()
        self.formde_proxy = PathToServices.DATA_ENGINE
        self.backfill_api_path = f'{self.formde_proxy}{StageDataEndPoints.api_back_fill_data}'

    async def create_template(self, template_file, data, user_id):
        try:
            template_name, template_id = data.get("template_name", ""), data.get("template_id", "")
            check_template = self.step_template.find_template(
                template_name=template_name)
            if not template_id:
                if check_template:
                    raise DuplicateTemplateNameError
                template_id = 'step_template_' + self.common_utils.get_next_id("step_template_id")
                meta = self.common_utils.get_user_meta(user_id, check_flag=True)
            else:
                if check_template and check_template.template_id != template_id:
                    raise DuplicateTemplateNameError
                template_data = self.step_template.find_by_id(template_id=template_id)
                meta = template_data.meta
                meta.update(self.common_utils.get_user_meta(user_id, check_flag=False))
            folder_name = os.path.join(PathToStorage.TEMPLATES_UPLOADS, TemplateStorage.templates_files)
            file_save_path = self.save_file_data(template_file, folder_name, template_name)
            final_data = dict(template_id=template_id, template_name=template_name,
                              logbook_id=data.get("logbook_id", ""),
                              associated_workflow_version=data.get("associated_workflow_version", 1),
                              meta=meta,
                              file_path=file_save_path)
            self.step_template.update_template_data(template_id=template_id,
                                                    project_id=data.get("project_id"), data=final_data, upsert=True)
            return True
        except Exception as e:
            logger.error(e.args)
            raise

    async def list_template_info(self, input_json: TemplateListRequest):
        try:
            response_data = TemplateListResponse()
            template_data, response_data.total_no = await self.list_paginated_template_info(input_json.dict())

            if response_data.total_no <= input_json.endRow - input_json.startRow:
                response_data.endOfRecords = True
            logbook_data = self.logbook_conn.get_logbook_data_by_aggregate(
                self.logbook_aggregate.logbook_key_values_list(input_json.project_id))
            logbook_data = logbook_data[0] if bool(logbook_data) else {}
            logbook_version_names_mapping = self.logbook_conn.get_logbook_data_by_aggregate(
                self.logbook_aggregate.get_logbook_versions_names_mapping(input_json.project_id))
            logbook_version_names_mapping = logbook_version_names_mapping[0] if logbook_version_names_mapping else {}
            all_users_list = list(
                self.user.users_list_by_aggregate(query=self.user_aggregate.get_users_list(input_json.project_id)))

            all_users_list = all_users_list[0] if bool(all_users_list) else {}
            final_template_data = []
            for each_template in template_data:
                if not logbook_data.get(each_template.get("logbook_id", ""), ""):
                    continue
                updated_on = each_template.get("updated_on", 0)
                updated_on = int(updated_on / 1000)
                each_template["updated_on"] = self.common_utils.get_iso_format(updated_on, input_json.timezone,
                                                                               "%d %B %Y")

                each_template["updated_by"] = all_users_list.get(each_template.get("updated_by", ""))

                each_template["logbook_name"] = logbook_data.get(each_template.get("logbook_id", ""), "")
                each_template["version_name"] = logbook_version_names_mapping.get(
                    f'{each_template.get("logbook_id", "")}${each_template.get("associated_workflow_version", 1) if each_template.get("associated_workflow_version", 1) else 1}')

                final_template_data.append(each_template)
            response_data.bodyContent = deepcopy(final_template_data)
            return response_data
        except Exception as e:
            logger.error(e.args)
            raise

    async def list_paginated_template_info(self, input_data):
        try:
            json_data = self.template_aggregate.list_template_info(input_data["project_id"])
            if input_data["filters"]:
                for key, value in input_data["filters"].items():
                    if bool(value):
                        key = key.lower()
                        if key == "updated_by":
                            key = "meta.updated_by"
                        if key == "updated_on":
                            key = "meta.updated_at"
                        if key.lower() != "meta.updated_at":
                            json_data[0]["$match"].update(
                                {key: {"$regex": re.escape(value), '$options': "i"}})
                        else:
                            json_data[0]["$match"].update(
                                {key: {"$gte": int(value[0]), "$lte": value[1]}})
            total_count_json = deepcopy(json_data)
            json_data.extend([{'$skip': input_data["startRow"]},
                              {'$limit': input_data["endRow"]}])
            response_aggregation = self.step_template.get_template_data_by_aggregate(json_data)
            if not response_aggregation:
                return list(), 0
            records = list(response_aggregation)
            total_records = list(self.step_template.get_template_data_by_aggregate(total_count_json))
            return records, len(total_records)
        except Exception as e:
            logger.error(f"Failed to list templates by filters: {e}")

    async def list_template_table_options(self, request_data: TemplateTableOptions):
        try:
            table_options_data = self.const_conn.find_constant(_type=request_data.type)
            return table_options_data.data
        except Exception as e:
            logger.error(f"Failed to list table_options_data of templates: {e}")

    async def delete_template(self, request_data: DeleteTemplate):
        try:
            template_data = self.step_template.find_by_id(template_id=request_data.template_id)
            if not os.path.exists(template_data.file_path):
                raise FileNotFoundError
            self.delete_existing_file(template_data.file_path)
            self.step_template.delete_template(template_id=request_data.template_id)
            return True
        except Exception as e:
            logger.error(e.args)
            raise

    async def fetch_instance(self, request_data: FetchTemplate):
        try:
            response_data = self.step_template.find_by_id(template_id=request_data.template_id)
            return response_data
        except Exception as e:
            logger.error(e.args)
            raise

    @staticmethod
    def save_file_data(template_file, folder_name, template_name, extension=False):
        try:
            out_file_path = os.path.join(folder_name)
            if not os.path.exists(out_file_path):
                os.makedirs(out_file_path)
            if not extension:
                file_extension = template_file.filename.split(".")[-1]
                template_name = f"{template_name}.{file_extension}"
            out_file_path = os.path.join(out_file_path, template_name)
            with open(out_file_path, 'wb') as out_file:
                content = template_file.file.read()
                out_file.write(content)
            return out_file_path
        except Exception as e:
            logger.error(e.args)
            raise

    @staticmethod
    def delete_existing_file(file_save_path):
        try:
            if os.path.exists(file_save_path):
                os.remove(file_save_path)
        except Exception as e:
            logger.error(e.args)
            raise

    async def download_template(self, template_id):
        try:
            template_data = self.step_template.find_by_id(template_id=template_id)
            if not os.path.exists(template_data.file_path):
                raise FileNotFoundError
            filepath = template_data.file_path
            filename = filepath.split("/")[-1]
            logger.debug(f"template downloaded : {filepath}")
            if not filepath:
                return None
            return filepath, filename
        except Exception as e:
            logger.exception(e)
            raise

    async def get_template_key_value_pairs(self, input_json: TemplateKeyValuePairs):
        try:
            list_data = self.step_template.get_steps_data_data_by_aggregate(
                self.template_aggregate.template_key_values_list(input_json.project_id, input_json.logbook_id))
            list_data = list_data[0] if bool(list_data) else dict()
            template_list = [{"label": key, "value": value} for value, key in list_data.items()]
            return template_list
        except Exception as e:
            logger.exception(e)
            raise

    def save_template_data_file(self, data_file, data, user_id, project_id, request_obj: Request):
        final_data, file_id = {}, ""
        try:
            template_data = self.step_template.find_by_id(template_id=data.get("template_id", ""))
            meta = self.common_utils.get_user_meta(user_id, check_flag=True)
            folder_name = os.path.join(PathToStorage.TEMPLATES_UPLOADS, TemplateStorage.upload_data_files)
            if not os.path.exists(folder_name):
                os.makedirs(folder_name)
            file_id = f'file_{self.common_utils.get_next_id("file_")}_{data_file.filename}'
            file_save_path = self.save_file_data(template_file=data_file, folder_name=folder_name,
                                                 template_name=file_id, extension=True)
            final_data |= dict(file_id=file_id, file_name=data_file.filename, template_id=template_data.template_id,
                               logbook_id=template_data.logbook_id,
                               associated_workflow_version=template_data.associated_workflow_version, meta=meta,
                               status="Success")
            self.step_data_files_conn.update_data_file(file_id=file_id, project_id=data.get("project_id"),
                                                       data=final_data, upsert=True)
            self.parse_file(file_save_path, data.get("date"), template_data.file_path, data.get("logbook_id"),
                            data.get("timezone", "Asia/Kolkata"), project_id, request_obj=request_obj)
            return True
        except Exception as e:
            final_data['status'] = "Failed"
            self.step_data_files_conn.update_data_file(file_id=file_id, project_id=data.get("project_id"),
                                                       data=final_data, upsert=True)
            logger.exception(e)
            raise

    @staticmethod
    def check_columns_match(template_file_path, sheet1_columns, sheet2_columns):
        try:
            wb = openpyxl.load_workbook(template_file_path)
            xls = pd.ExcelFile(template_file_path, engine='openpyxl')
            template_sheets = wb.get_sheet_names()
            template_sheet1_columns = list(pd.read_excel(xls, template_sheets[0]).keys())
            template_sheet2_columns = list(pd.read_excel(xls, template_sheets[1]).keys())
            sheet1_columns.sort(), template_sheet1_columns.sort(), sheet2_columns.sort(), template_sheet2_columns.sort()
            if sheet1_columns == template_sheet1_columns and \
                    sheet2_columns == template_sheet2_columns:
                return True
            return False
        except Exception as e:
            logger.exception(e)
            raise

    def parse_file(self, file_save_path, date, template_file_path, logbook_id, timezone, project_id,
                   request_obj: Request):
        try:
            if not file_save_path.endswith('.xlsx'):
                raise BulkUploadError
            wb = openpyxl.load_workbook(file_save_path)
            sheets_name = wb.get_sheet_names()
            xls = pd.ExcelFile(file_save_path, engine='openpyxl')
            df1 = pd.read_excel(xls, sheets_name[0])
            records = df1.to_dict('records')
            df2 = pd.read_excel(xls, sheets_name[1])
            df2 = df2.dropna()
            df2_columns = list(df2.keys())
            columns_match = self.check_columns_match(template_file_path, list(df1.keys()), df2_columns)
            if not columns_match:
                raise ColumnsMisMatch
            key_column = df2_columns[0]
            value_column = df2_columns[1]
            column_param_dict = df2.set_index(key_column)[value_column].to_dict()
            column_param_dict_lower = {key.lower(): value for key, value in column_param_dict.items()}
            tag_dict, site_dict = {}, {}
            for each in records:
                tag_hierarchy = each['iLens Hierarchy']
                for key, value in each.items():
                    key = key.lower()
                    if all((
                            key in column_param_dict_lower, not pd.isna(value),
                            not pd.isna(tag_hierarchy))):
                        if not isinstance(value, (int, float)):
                            raise InvalidValueFound(f"Invalid value {value} in column {key}")
                        if '$' not in column_param_dict_lower[key]:
                            tag_hierarchy = f"{tag_hierarchy}${column_param_dict_lower[key]}"
                        else:
                            tag_hierarchy = tag_hierarchy + column_param_dict_lower[key]
                        tag_dict.update({tag_hierarchy: value})
                        if site_dict.get(tag_hierarchy.split("$")[0]):
                            site_dict[tag_hierarchy.split("$")[0]].update({tag_hierarchy: value})
                        else:
                            site_dict.update({tag_hierarchy.split("$")[0]: {tag_hierarchy: value}})
            for key, values in site_dict.items():
                kairos_dict = {date: values}
                self.common_utils.publish_data_to_kafka(kairos_dict, project_id)
            back_fill_data = {
                "logbook_ids": [logbook_id],
                "values": tag_dict,
                "tz": timezone,
                "time_str": self.common_utils.get_iso_format(date / 1000, timezone=timezone,
                                                             timeformat=date_time_with_hour)
            }
            cookies = request_obj.cookies
            headers = {
                'login-token': request_obj.headers.get('login-token', request_obj.cookies.get('login-token')),
                'projectId': request_obj.cookies.get("projectId", request_obj.cookies.get("project_id",
                                                                                          request_obj.headers.get(
                                                                                              "projectId"))),
                'userId': request_obj.cookies.get("user_id",
                                                  request_obj.cookies.get("userId", request_obj.headers.get(
                                                      "userId")))}
            resp = requests.post(url=self.backfill_api_path, json=back_fill_data, cookies=cookies, headers=headers)
            if resp.status_code not in CommonStatusCode.SUCCESS_CODES:
                logger.debug('Failed response from back fill api')
            else:
                logger.info('Back fill api successfully executed')
        except Exception as e:
            logger.exception(e)
            raise

    async def list_uploaded_files(self, input_json: UploadedFileList):
        try:
            response_data = UploadedFileListResponse()
            files_list_data, response_data.total_no = await self.list_paginated_files_list_info(
                input_json.dict())
            if response_data.total_no <= input_json.endRow - input_json.startRow:
                response_data.endOfRecords = True
            logbook_data = self.logbook_conn.get_logbook_data_by_aggregate(
                self.logbook_aggregate.logbook_key_values_list(input_json.project_id))
            logbook_data = logbook_data[0] if logbook_data else dict()
            logbook_version_names_mapping = self.logbook_conn.get_logbook_data_by_aggregate(
                self.logbook_aggregate.get_logbook_versions_names_mapping(input_json.project_id))
            logbook_version_names_mapping = logbook_version_names_mapping[0] if logbook_version_names_mapping else {}
            all_users_list = list(
                self.user.users_list_by_aggregate(query=self.user_aggregate.get_users_list(input_json.project_id)))
            if bool(all_users_list):
                all_users_list = all_users_list[0]
            else:
                all_users_list = dict()
            file_data = []
            for each_template in files_list_data:
                if not logbook_data.get(each_template.get("logbook_id", ""), ""):
                    continue
                updated_on = each_template.get("updated_on", 0)
                updated_on = int(updated_on / 1000)
                each_template["updated_on"] = self.common_utils.get_iso_format(updated_on, input_json.timezone,
                                                                               "%d %B %Y")
                each_template["updated_by"] = all_users_list.get(each_template.get("updated_by", ""))
                each_template["logbook_name"] = logbook_data.get(each_template.get("logbook_id", ""), "")
                each_template['version_name'] = logbook_version_names_mapping.get(
                    f'{each_template.get("logbook_id", "")}${each_template.get("associated_workflow_version", 1) if each_template.get("associated_workflow_version", 1) else 1}')
                file_data.append(each_template)
            response_data.bodyContent = deepcopy(file_data)
            return response_data

        except Exception as e:
            logger.error(e.args)
            raise

    async def list_paginated_files_list_info(self, input_data):
        try:
            json_data = self.step_data_files_agg.list_step_data_files(input_data["project_id"])
            if input_data["filters"]:
                for key, value in input_data["filters"].items():
                    if bool(value):
                        key = key.lower()
                        if key == "updated_by":
                            key = "meta.updated_by"
                        if key == "updated_on":
                            key = "meta.updated_at"
                        if key.lower() != "meta.updated_at":
                            json_data[0]["$match"].update(
                                {key: {"$regex": re.escape(value), '$options': "i"}})
                        else:
                            json_data[0]["$match"].update(
                                {key: {"$gte": int(value[0]), "$lte": value[1]}})
            total_count_json = deepcopy(json_data)
            json_data.extend([{'$skip': input_data["startRow"]},
                              {'$limit': input_data["endRow"]}])
            response_aggregation = self.step_data_files_conn.get_step_files_data_by_aggregate(json_data)
            if not response_aggregation:
                return list(), 0
            records = list(response_aggregation)
            total_records = list(self.step_data_files_conn.get_step_files_data_by_aggregate(total_count_json))
            return records, len(total_records)
        except Exception as e:
            logger.error(f"Failed to list templates by filters: {e}")

    async def download_data_file(self, file_id):
        try:
            folder_name = os.path.join(PathToStorage.TEMPLATES_UPLOADS, TemplateStorage.upload_data_files)
            step_file_data = self.step_data_files_conn.find_by_id(file_id=file_id)
            if not step_file_data:
                raise FileNotFoundError
            filepath = f"{folder_name}/{step_file_data.file_id}"
            filename = step_file_data.file_name
            logger.debug(f"template downloaded : {filepath}")
            if not filepath:
                return None
            return filepath, filename
        except Exception as e:
            logger.exception(e)
            raise

    async def delete_data_file(self, request_data: DeleteDataFile):
        try:
            folder_name = os.path.join(PathToStorage.TEMPLATES_UPLOADS, TemplateStorage.upload_data_files)
            step_file_data = self.step_data_files_conn.find_by_id(file_id=request_data.file_id)
            filepath = f"{folder_name}/{step_file_data.file_id}"
            if not os.path.exists(filepath):
                raise FileNotFoundError
            self.delete_existing_file(filepath)
            self.step_data_files_conn.delete_data_file(file_id=request_data.file_id)
            return True
        except Exception as e:
            logger.error(e.args)
            raise
