import time
from datetime import datetime

import pytz

from scripts.core.schemas.stages import SaveRemarks, FetchRemarks
from scripts.db import TaskInstanceData, mongo_client
from scripts.db.mongo.ilens_assistant.collections.project_remarks import ProjectRemarks
from scripts.db.mongo.ilens_configuration.collections.tags import Tag
from scripts.logging.logging import logger
from scripts.utils.common_utils import CommonUtils


class RemarkHandler:
    def __init__(self, project_id=None):
        self.stage_conn = TaskInstanceData(mongo_client=mongo_client, project_id=project_id)
        self.tag_conn = Tag(mongo_client=mongo_client, project_id=project_id)

        self.project_info_conn = ProjectRemarks(mongo_client=mongo_client, project_id=project_id)
        self.common_utils = CommonUtils(project_id=project_id)

    def save_remarks_data(self, request_data: SaveRemarks, user_id):
        try:
            if request_data.stage_id:
                stage_record = self.stage_conn.find_by_id(stage_id=request_data.stage_id)
                if bool(stage_record):
                    stage_remarks = stage_record.remarks
                    user_name = self.common_utils.get_user_name_from_id(user_id)
                    request_data.updated_on = int(time.time() * 1000)
                    request_data.user_id = user_id
                    request_data.user_name = user_name
                    if request_data.type == "parameter_specific":
                        tag_name = self.tag_conn.find_name_by_id(
                            tag_id=request_data.parameter_specific.get("parameter"))
                        request_data.parameter_specific["parameter_id"] = request_data.parameter_specific.get(
                            "parameter")
                        request_data.parameter_specific["parameter"] = tag_name
                    stage_remarks.append(request_data.dict(exclude_none=True))
                    self.stage_conn.update_stage_data(stage_id=request_data.stage_id, data={"remarks": stage_remarks})
            elif request_data.project_id:
                project_record = self.project_info_conn.find_by_id(project_id=request_data.project_id)
                if not project_record:
                    project_record = dict()

                global_remarks = project_record.get("remarks", list())
                user_name = self.common_utils.get_user_name_from_id(user_id)
                request_data.updated_on = int(time.time() * 1000)
                request_data.user_id = user_id
                request_data.user_name = user_name
                if request_data.type == "parameter_specific":
                    tag_name = self.tag_conn.find_name_by_id(
                        tag_id=request_data.parameter_specific.get("parameter"))
                    request_data.parameter_specific["parameter_id"] = request_data.parameter_specific.get(
                        "parameter")
                    request_data.parameter_specific["parameter"] = tag_name
                global_remarks.append(request_data.dict(exclude_none=True))
                self.project_info_conn.update_project_info(request_data.project_id, global_remarks, upsert=True)
            return True
        except Exception as e:
            logger.error(f"Error occurred in fetch stage stepper {e}")
            raise

    def fetch_remarks(self, request_data: FetchRemarks):
        try:
            if request_data.stage_id:
                stage_record = self.stage_conn.find_by_id(stage_id=request_data.stage_id)
                if not stage_record.remarks:
                    return dict()
                for each in stage_record.remarks:
                    time_convert = datetime.fromtimestamp(each["updated_on"] / 1000,
                                                          tz=pytz.timezone(request_data.timezone
                                                                           )). \
                        strftime("%d %b %y, %H:%M")
                    each["updated_on"] = time_convert
                    if each.get("type") == "parameter_specific" and bool(
                            each.get("parameter_specific", {}).get("observed_time")):
                        time_convert = datetime.fromtimestamp(
                            each.get("parameter_specific", {}).get("observed_time") / 1000,
                            tz=pytz.timezone(request_data.timezone
                                             )). \
                            strftime("%d %b %y, %H:%M")
                        each["parameter_specific"]["observed_time"] = time_convert

                response = dict(comments=stage_record.remarks)
                return response
            elif request_data.project_id:
                global_remarks = self.project_info_conn.find_by_id(project_id=request_data.project_id)
                if not global_remarks:
                    return dict()
                for each in global_remarks["remarks"]:
                    time_convert = datetime.fromtimestamp(each["updated_on"] / 1000,
                                                          tz=pytz.timezone(request_data.timezone
                                                                           )). \
                        strftime("%d %b %y, %H:%M")
                    each["updated_on"] = time_convert
                    if each.get("type") == "parameter_specific" and bool(
                            each.get("parameter_specific", {}).get("observed_time")):
                        time_convert = datetime.fromtimestamp(
                            each.get("parameter_specific", {}).get("observed_time") / 1000,
                            tz=pytz.timezone(request_data.timezone
                                             )). \
                            strftime("%d %b %y, %H:%M")
                        each["parameter_specific"]["observed_time"] = time_convert
                response = dict(comments=global_remarks["remarks"])
                return response
        except Exception as e:
            logger.error(f"Error occurred in fetch stage stepper {e}")
            raise
