import pandas as pd
import requests
from fastapi import Request
from scripts.config.app_configurations import PathToServices
from scripts.constants import StatusCodes
from scripts.constants.api import DataEngineEndpoints
from scripts.core.engine.custom_implementations import CustomImplementations
from scripts.core.schemas.forms import SaveForm, TasksInfoList
from scripts.db import PeriodicData, TaskInstanceData, TaskInstance
from scripts.db import mongo_client
from scripts.logging.logging import logger
from scripts.utils.common_utils import CommonUtils
from scripts.utils.data_processor import ProcessData


class DataEngine:
    def __init__(self, project_id=None):
        self.common_utils = CommonUtils(project_id=project_id)
        self.periodic_conn = PeriodicData(mongo_client, project_id=project_id)
        self.tasks_data = TaskInstanceData(mongo_client, project_id=project_id)
        self.tasks = TaskInstance(mongo_client, project_id=project_id)
        self.processor = ProcessData(project_id=project_id)
        self.custom_imp = CustomImplementations(project_id=project_id)

    def get_iot_param(self, form_info, form_data, date, tz, request_obj: Request):
        try:
            tag_dict = dict()
            for _name, props in form_info.items():
                current_properties = set(props.keys())

                if _name not in form_data and {"time", "time_associated", "tag", "manual_entry"}.intersection(
                        current_properties) in [{"tag"}, {"tag", "manual_entry"}]:
                    tag_dict.update({_name: props["tag"]})
            if len(tag_dict) == 0:
                return form_data
            iot_values = self.get_tag_values(set(tag_dict.values()), for_date=date, tz=tz, request_obj=request_obj)
            if not iot_values:
                return form_data
            returned_data = {
                _name: round(iot_values[tag], 2) if isinstance(iot_values[tag], (int, float)) else iot_values[tag] for
                _name, tag in tag_dict.items() if
                tag in iot_values.keys()}
            form_data.update(returned_data)
            return form_data
        except Exception as e:
            logger.error("Failed in get_iot_param", e)

    def data_for_next_day(self, date, step_id, form_df, property_name, relative_day=1):
        try:
            if property_name not in form_df:
                return list()
            next_date_in_format = self.common_utils.get_next_date(date, "yyyy-MM-dd", relative_day)
            next_record = self.periodic_conn.find_by_date_and_step(next_date_in_format, step_id)
            if not next_record.data:
                return list()
            return next_record.data
        except Exception as e:
            logger.error("Failed in data_for_next_day", e)

    def get_last_value(self, step_id, today, form_df):
        try:
            if "default" not in form_df:
                return []
            form_df = form_df[form_df['default'] == "last_value"]
            default_tags = list(form_df["tag"]) if "tag" in form_df.columns else []
            last_record = self.periodic_conn.find_by_latest_data(step_id, today, default_tags)
            if not last_record:
                return []
            return [last_record]
        except Exception as e:
            logger.error("Failed in get_last_value", e)
            return []

    def get_data_for_date(self, request_data: SaveForm, step_id, field_props, date, datetime_obj):
        try:
            if not field_props:
                return dict()
            form_df = pd.DataFrame.from_dict(field_props, orient='index')
            current_record = self.periodic_conn.find_by_date_and_step(date, step_id)
            prv_record = self.data_for_next_day(date, step_id, form_df, "previous_day", -1)
            next_record = self.data_for_next_day(date, step_id, form_df, "next_day", 1)
            latest_record = self.get_last_value(step_id, datetime_obj, form_df)
            month_to_date = self.custom_imp.month_to_date(step_id, datetime_obj, form_df)
            current_record.manual_data.update(month_to_date)
            if not any([current_record.data, current_record.manual_data, prv_record, next_record, latest_record]):
                return dict()
            if not any([current_record.data, prv_record, next_record, latest_record]) and current_record.manual_data:
                return current_record.manual_data
            if all(["time_associated" not in form_df, latest_record, current_record.manual_data]):
                return current_record.manual_data
            if "time_associated" in form_df:
                field_props = self.custom_imp.time_associated(form_df, current_record.data, request_data, next_record,
                                                              prv_record, latest_record)
                field_props.update(current_record.manual_data)
                return field_props
        except Exception as e:
            logger.error("Failed in get_data_for_date", e)
            raise

    def get_current_and_next_df(self, request_data: SaveForm, field_props, next_record, record):
        try:
            base_df = pd.DataFrame(columns=["time_associated", "time", "tag", "next_day", "values", "prop", "manual"])
            form_df = pd.DataFrame.from_dict(field_props, orient='index')
            if not record.data:
                return base_df, base_df, form_df, dict()
            data_req = record.data
            if "time_associated" in form_df:
                current, next_day = self.custom_imp.get_data_dfs(form_df, data_req, request_data, next_record)
                return current, next_day, form_df, record.manual_data
        except Exception as e:
            logger.error("Failed in get_data_for_date", e)
            raise

    @staticmethod
    def get_tag_values(tag_list, request_obj: Request,
                       ignore_stale=False,
                       for_date=None,
                       last_data=True,
                       tz="Asia/Kolkata",
                       from_time=None,
                       to_time=None):
        try:
            if not tag_list:
                return None
            cookies = request_obj.cookies
            tag_list = [x for x in tag_list if x]
            tag_json = dict(tag_list=list(tag_list),
                            tz=tz)
            if for_date:
                tag_json.update(filter_by_date=for_date)
            elif all([from_time, to_time, not last_data]):
                tag_json.update(from_time=from_time, to_time=to_time)
            response = requests.post(
                f"{PathToServices.DATA_ENGINE}{DataEngineEndpoints.api_iot_param}"
                f"?last_data={last_data}&ignore_stale={ignore_stale}", json=tag_json,
                timeout=30, cookies=cookies)
            status = response.status_code
            if status == 404:
                raise ModuleNotFoundError
            content = response.json()
            if "data" not in content or not content["data"]:
                return None
            elif status not in StatusCodes.SUCCESS or content["status"] != "success" or not content["data"]:
                logger.debug(f"Content returned: {content}")
                logger.error("Error Encountered: Communication to Data engine was unsuccessful")
                return None
            logger.info("Communication to Data engine was successful, response content: ", content)
            values = content["data"]["values"]
            if isinstance(values, list):
                values = values[0]
            return values
        except requests.exceptions.ReadTimeout:
            raise TimeoutError(f"Request Time out on IOT param call")
        except Exception as e:
            logger.error(f"Error Encountered while contacting Data Engine, {e}")
            raise

    def get_tasks_from_logbooks(self, logbook_list):
        try:
            tasks_list = self.tasks.find_by_logbooks(logbook_list)
            return [TasksInfoList(**x) for x in tasks_list]
        except Exception as e:
            logger.error(f"Error Encountered in get_tasks_from_logbooks, {e}")
            raise

    def get_data_for_task(self, task_id):
        try:
            stage_list = self.tasks_data.find_data_by_task_id(task_id)
            return stage_list
        except Exception as e:
            logger.error(f"Error Encountered in get_tasks_from_logbooks, {e}")
            raise
