import os
import time
from datetime import datetime

import httpx
import pytz

from scripts.config.app_configurations import PathToServices
from scripts.constants.api import OeeServicesEndpoints
from scripts.constants.app_constants import CommonStatusCode
from scripts.constants.date_constants import ui_time_format_data
from scripts.core.engine.task_engine import TaskEngine
from scripts.core.schemas.forms import CustomActionsModel
from scripts.db import mongo_client, TaskInstance
from scripts.db.mongo.ilens_assistant.collections.logbook import LogbookInfo
from scripts.db.mongo.ilens_configuration.aggregations.config_aggregate import ConfigAggregate
from scripts.db.mongo.ilens_configuration.collections.customer_projects import CustomerProjects
from scripts.errors import InternalError
from scripts.logging.logging import logger
from scripts.utils.common_utils import CommonUtils


class CustomAction:
    def __init__(self, custom_action: CustomActionsModel):
        self.custom_model: CustomActionsModel = custom_action
        self.customer_projects_con = CustomerProjects(mongo_client=mongo_client)
        self.config_aggregate = ConfigAggregate()
        self.task_engine = TaskEngine(project_id=self.custom_model.project_id)
        self.task_inst_conn = TaskInstance(mongo_client, project_id=custom_action.project_id)
        self.logbook_conn = LogbookInfo(mongo_client=mongo_client, project_id=custom_action.project_id)
        self.common_utils = CommonUtils()
        self.create_batch_api = f"{PathToServices.OEE_SERVICES}{OeeServicesEndpoints.api_create_batch}"

    def trigger_action(self):
        try:
            site_templates = self.customer_projects_con.get_project_data_by_aggregate(
                self.config_aggregate.get_project_template(self.custom_model.project_id))
            site_templates = site_templates[0].get("data") if bool(site_templates) else []
            hierarchy_id_str = ""
            task_data = self.task_inst_conn.find_by_task_id(task_id=self.custom_model.task_details.task_id)
            logbook_data = self.logbook_conn.find_by_id(task_data.logbook_id)
            if hierarchy := self.task_engine.get_hierarchy(logbook_data.dict(), task_data.dict()):
                hierarchy_id_str = self.task_engine.get_hierarchy_string(hierarchy, site_templates)
            task_creation_time = task_data.meta.get("created_at")
            task_completion_time = task_data.meta.get("completed_at")
            start_property_name = os.environ.get("OEE_START_TIME_KEY", default="oee_start_time")
            end_property_name = os.environ.get("OEE_END_TIME_KEY", default="oee_end_time")
            prod_start_time = self.common_utils.get_task_time(task_time=task_creation_time,
                                                              custom_model=self.custom_model,
                                                              task_property_name=start_property_name)

            prod_end_time = self.common_utils.get_task_time(task_time=task_completion_time,
                                                            custom_model=self.custom_model,
                                                            task_property_name=end_property_name,
                                                            task_type="complete")
            if not prod_end_time:
                prod_start_time = task_completion_time / 1000 if task_completion_time else time.time()
            if prod_end_time > prod_start_time:
                prod_end_time = datetime.now(tz=pytz.timezone(self.custom_model.tz))
            prod_end_time = self.common_utils.get_iso_format(
                timestamp=int(prod_end_time.timestamp()),
                timezone=self.custom_model.tz,
                timeformat=ui_time_format_data["yyyy-MM-dd HH:mm:ss"])
            prod_start_time = self.common_utils.get_iso_format(timestamp=int(prod_start_time.timestamp()),
                                                               timezone=self.custom_model.tz,
                                                               timeformat=ui_time_format_data["yyyy-MM-dd HH:mm:ss"])
            payload = dict(reference_id=task_data.reference_id,
                           hierarchy=hierarchy_id_str,
                           project_id=self.custom_model.project_id,
                           tz=self.custom_model.tz,
                           prod_start_time=prod_start_time,
                           prod_end_time=prod_end_time)
            with httpx.Client() as client:
                resp = client.post(url=self.create_batch_api, cookies=self.custom_model.request_obj.cookies,
                                   json=payload, timeout=15)
                logger.debug(f"Resp Code:{resp.status_code}")
                if resp.status_code not in CommonStatusCode.SUCCESS_CODES:
                    logger.error(f"Failed while calling custom API: {resp.status_code}")
                    # raise InternalError(f"API not callable: Status - {resp.status_code}")
                if resp.headers.get('Content-Type').startswith('application/json') or resp.headers.get(
                        'content-type').startswith('application/json'):
                    message = resp.json()
                else:
                    message = dict(message="Unable to decode response, API Triggered")
                return True, message.get("message", "Batch Created successfully")
        except InternalError:
            raise
        except Exception as e:
            logger.error(f"Exception occurred in while finishing batch: {e}")
