"""
This file exposes configurations from config file and environments as Class Objects
"""
import shutil

if __name__ == '__main__':
    from dotenv import load_dotenv

    load_dotenv()
import os.path
import sys
from configparser import ConfigParser, BasicInterpolation


class EnvInterpolation(BasicInterpolation):
    """
    Interpolation which expands environment variables in values.
    """

    def before_get(self, parser, section, option, value, defaults):
        value = super().before_get(parser, section, option, value, defaults)

        if not os.path.expandvars(value).startswith('$'):
            return os.path.expandvars(value)
        else:
            return


try:
    config = ConfigParser(interpolation=EnvInterpolation())
    config.read("conf/application.conf")
except Exception as e:
    print(f"Error while loading the config: {e}")
    print("Failed to Load Configuration. Exiting!!!")
    sys.stdout.flush()
    sys.exit()


class Service:
    MODULE_NAME = config["MODULE"]["name"]
    HOST = config.get("SERVICE", "host")
    PORT = config.getint("SERVICE", "port")
    secure_cookie = config.getboolean("SERVICE", "secure_cookie", fallback=True)
    verify_signature = os.environ.get("VERIFY_SIGNATURE", False)
    protected_hosts = os.environ.get("PROTECTED_HOSTS", "").split(",")


class DBConf:
    MONGO_URI = config.get('MONGO_DB', 'uri')
    if not MONGO_URI:
        print("Error, environment variable MONGO_URI not set")
        sys.exit(1)
    MAINTENANCE_DB_URI = config.get('POSTGRES', "maintenance")
    if not MAINTENANCE_DB_URI:
        print("MAINTENANCE_DB_URI env variables missing")
        sys.exit(1)
    ASSISTANT_DB_URI = config.get('POSTGRES', "assistant")
    if not ASSISTANT_DB_URI:
        print("ASSISTANT_DB_URI env variables missing")
        sys.exit(1)


class KafkaConf:
    host = config.get('KAFKA', 'host')
    port = config.get('KAFKA', 'port')
    topic = config.get('KAFKA', 'topic')
    backdated_topic = config.get('KAFKA', 'history_topic')
    if not any([topic, host, port]):
        print("KAFKA env variables missing, continuing without Kafka/Kairos support")
    audit_topic = config.get('KAFKA', 'audit_topic')
    enable_sites_partition = config.getboolean("KAFKA", "ENABLE_KAFKA_PARTITION", fallback=True)

    split_key = config["KAFKA"].get('KAFKA_PARTITION_KEY', 'site_id')
    round_robin_enable = config.getboolean("KAFKA", "ROUND_ROBIN_PARTITION", fallback=True)

    redis_db = config.getint("KAFKA", "partition_db")


class Logging:
    level = config.get("LOGGING", "level", fallback="INFO")
    level = level or "INFO"
    print(f"Logging Level set to: {level}")


class StoragePaths:
    module_name = config.get('SERVICE', 'module_name')
    if not module_name:
        module_name = "form_management"
    base_path = os.path.join("data", module_name)


class DatabaseConstants:
    metadata_db = config.get("DATABASES", "metadata_db")
    if not bool(metadata_db):
        metadata_db = "ilens_configuration"
    ilens_assistant_db = config.get("DATABASES", "ilens_assistant")
    if not bool(ilens_assistant_db):
        ilens_assistant_db = "ilens_assistant"
    ilens_asset_model_db = config.get("DATABASES", "ilens_asset_model")
    if not bool(ilens_asset_model_db):
        ilens_asset_model_db = "ilens_asset_model"


class PathToServices:
    DATA_ENGINE = config.get("PATH_TO_SERVICES", "data_engine")
    if not bool(DATA_ENGINE):
        print("FORM_DE not set, proceeding without data engine support")
    METADATA_SERVICES = config.get("PATH_TO_SERVICES", "metadata_services")
    if not bool(METADATA_SERVICES):
        print("METADATA_SERVICES not set, proceeding without metadata_services support")

    AUDIT_PROXY = config.get("PATH_TO_SERVICES", "audit_proxy")
    if not bool(AUDIT_PROXY):
        print("AUDIT_PROXY not set, proceeding without audit_proxy support")
    WORKFLOW_MT = config.get("PATH_TO_SERVICES", "workflow_mt")
    if not bool(WORKFLOW_MT):
        print("WORKFLOW_MT_PROXY not set, proceeding without audit_proxy support")
    FORM_MT = config.get("PATH_TO_SERVICES", "form_mt")
    if not bool(FORM_MT):
        print("Error, environment variable FORM_MT not set")
        sys.exit(1)
    ILENS_EVENTS = config.get("PATH_TO_SERVICES", "ilens_events")
    if not bool(ILENS_EVENTS):
        print("Error, environment variable ILENS_EVENTS not set")
        sys.exit(1)
    OEE_SERVICES = config.get("PATH_TO_SERVICES", "oee_services")
    if not bool(OEE_SERVICES):
        print("Error, environment variable OEE_SERVICES not set")
        sys.exit(1)


class PathToStorage:
    BASE_PATH = config.get("DIRECTORY", "base_path")
    if not BASE_PATH:
        print("Error, environment variable BASE_PATH not set")
        sys.exit(1)
    MOUNT_DIR = config.get("DIRECTORY", "mount_dir")
    if not MOUNT_DIR:
        print("Error, environment variable MOUNT_DIR not set")
        sys.exit(1)
    MODULE_PATH = os.path.join(BASE_PATH, MOUNT_DIR.lstrip('/'))
    FORM_IO_UPLOADS = os.path.join(MODULE_PATH, "form_io_uploads")
    TEMPLATES_UPLOADS = os.path.join(MODULE_PATH, "templates_uploads")
    LOGS_MODULE_PATH = f"{BASE_PATH}/logs{MOUNT_DIR}/"


class KeyPath:
    keys_path = config['DIRECTORY']['keys_path']
    if not os.path.isfile(os.path.join(keys_path, "public")) or not os.path.isfile(os.path.join(keys_path, "private")):
        if not os.path.exists(keys_path):
            os.makedirs(keys_path)
        shutil.copy(os.path.join("assets", "keys", "public"), os.path.join(keys_path, "public"))

        shutil.copy(os.path.join("assets", "keys", "private"), os.path.join(keys_path, "private"))

    public = os.path.join(keys_path, "public")
    private = os.path.join(keys_path, "private")


class RedisConfig:
    uri = config.get("REDIS", "uri")
    login_db = config["REDIS"]["login_db"]
    project_tags_db = config.getint("REDIS", "project_tags_db")


class KairosConfig:
    uri = config.get("KAIROS", "kairos_url")


class BackFill:
    interval_in_mins = config.get("BACKFILL", "interval", fallback=60)
    trigger_step_threshold = config.getint("BACKFILL", "trigger_bandwidth", fallback=300)


class EnableAuditing:
    periodic_entry_auditing = config.getboolean("AUDITING", "periodic_entry_auditing", fallback=False)

    form_non_periodic_auditing = config.getboolean("AUDITING", "form_non_periodic_auditing", fallback=False)

    form_periodic_auditing = config.getboolean("AUDITING", "form_periodic_auditing", fallback=False)


class OtherService:
    EMAIL_URL = config["PATH_TO_OTHER_SERVICES"]["email_service"]


class MQTTConf:
    uri = config["MQTT"]["uri"]
    host = config["MQTT"]["host"]
    port = int(config["MQTT"]["port"])
    publish_base_topic = config["MQTT"]["publish_base_topic"]


class EnableEvents:
    enable_events = config.getboolean("EVENTS", "enable_events", fallback=True)
