import base64
import json
import math
import os
import re
import time
import traceback
from datetime import datetime, timedelta, timezone
from functools import lru_cache, wraps
from typing import List, Optional

import httpx
import pandas as pd
import pendulum
import pytz
from dateutil import parser

from tb_sdk.connectors.constants import CommonConstants, CommonKeys
from tb_sdk.connectors.constants.app_constants import CommonStatusCode
from tb_sdk.connectors.constants.secrets import Secrets
from tb_sdk.connectors.db.mongo import mongo_client
from tb_sdk.connectors.db.mongo.ilens_configuration.collections.user import User
from tb_sdk.connectors.db.mongo.ilens_configuration.collections.user_project import UserProject
from tb_sdk.connectors.schemas.meta import ExternRequest, MetaInfoSchema
from tb_sdk.connectors.utils.security_utils.apply_encryption_util import create_token
from tb_sdk.connectors.utils.security_utils.jwt_util import JWT


def timed_lru_cache(seconds: int = 10, maxsize: int = 128):
    def wrapper_cache(func):
        func = lru_cache(maxsize=maxsize)(func)
        func.lifetime = timedelta(seconds=seconds)
        func.expiration = datetime.now(timezone.utc) + func.lifetime

        @wraps(func)
        def wrapped_func(*args, **kwargs):
            if datetime.now(timezone.utc) >= func.expiration:
                print("Cache Expired")
                func.cache_clear()
                func.expiration = datetime.now(timezone.utc) + func.lifetime

            return func(*args, **kwargs)

        return wrapped_func

    return wrapper_cache


class CommonUtils(CommonKeys, CommonConstants):
    def __init__(self):
        self.user_conn = User(mongo_client)
        self.user_proj = UserProject(mongo_client)

    def get_user_roles_by_project_id(self, user_id, project_id):
        user_rec = self.user_conn.find_user_by_project_id(user_id=user_id, project_id=project_id)
        user_rec = user_rec if bool(user_rec) else {}
        if not user_rec:
            user_rec = self.user_proj.fetch_user_project(user_id=user_id, project_id=project_id)
            user_rec = user_rec if bool(user_rec) else {}
        return user_rec.get("userrole", [])

    @staticmethod
    def create_token(host: str = "127.0.0.1", user_id=None, internal_token=Secrets.token, project_id=None):
        """
        This method is to create a cookie
        """

        try:
            if user_id is None:
                user_id = "user_099"
            return create_token(user_id=user_id, ip=host, token=internal_token, project_id=project_id)
        except Exception as e:
            print(e)
            raise

    @staticmethod
    def hit_external_service(
        api_url,
        payload=None,
        request_cookies=None,
        timeout=int(os.environ.get("REQUEST_TIMEOUT", default=30)),
        method="post",
        params=None,
        auth=None,
        headers=None,
    ):
        try:
            print("Inside function to hit external services" f"\nURL - {api_url}")
            payload_json = ExternRequest(
                url=api_url, timeout=timeout, cookies=request_cookies, params=params, auth=auth, headers=headers
            )
            payload_json = payload_json.dict(exclude_none=True)
            if payload:
                payload_json.update(json=payload)
            with httpx.Client() as client:
                for _ in range(3):
                    method_type = getattr(client, method)
                    resp = method_type(**payload_json)
                    print(f"Resp Code:{resp.status_code}")
                    if resp.status_code in CommonStatusCode.SUCCESS_CODES:
                        return resp.json()
                    elif resp.status_code == 404:
                        print(f"Module not found: {api_url}")
                        raise ModuleNotFoundError
                    elif resp.status_code == 401:
                        print(f"Unauthorized to execute request on {api_url}")
                    print(
                        f"Resp Message:{resp.status_code} \n", f"Cookies: {request_cookies} \n", f"Rest API: {api_url}"
                    )
                time.sleep(3)
        except Exception as e:
            print(e)
            raise

    @staticmethod
    def encode_using_jwt(file_name=None):
        jwt = JWT()
        payload = {}
        if file_name:
            payload.update({"file_name": file_name})
        exp = datetime.now() + timedelta(minutes=30)
        _extras = {"iss": Secrets.issuer, "exp": exp}
        _payload = {**payload, **_extras}
        new_token = jwt.encode(_payload)
        return new_token

    @staticmethod
    def generate_cookie_from_user_project(user_id, project_id, ip="127.0.0.1"):
        """
        The generate_cookie_from_user_project function generates a cookie from the user_id and project_id.
            Args:
                user_id (int): The id of the user to generate a cookie for.
                project_id (int): The id of the project to generate a cookie for.
                ip (str, optional): Defaults to &quot;127.0.0.2&quot;. The IP address that will be used in generating this token/cookie.

        :param user_id: Create a token for the user
        :param project_id: Create a cookie that is specific to the project
        :param ip: Store the ip address of the user
        :return: A metainfoschema object
        """
        generated_new_token = create_token(user_id, ip, Secrets.token, project_id=project_id)

        return MetaInfoSchema(
            login_token=generated_new_token,
            projectId=project_id,
            project_id=project_id,
            user_id=user_id,
            ip_address=ip,
        )
