import json
import time
from datetime import datetime, timedelta
from typing import Any

import httpx
import pandas as pd
import pytz
import shortuuid

from tb_sdk.config import PathToServices
from tb_sdk.connectors.constants import APIEndPoints
from tb_sdk.connectors.errors import InputRequestError, RequiredFieldMissing, ErrorMessages

from tb_sdk.connectors.schemas.tasks_schema import SaveForm, FormRender, TaskDetails, TaskListSchema
from tb_sdk.connectors.utils.common_utils import CommonUtils


class Tasks:
    def __init__(self):
        self.form_render_url = f"{PathToServices.FORM_MT}{APIEndPoints.render_proxy}"
        self.workflow_task_url = f"{PathToServices.WORKFLOW_MT}{APIEndPoints.task_proxy}"

    def render_form(self, request_data: FormRender, user_id: str, project_id: str):

        """
        Info:
            The render_form function is used to render the form.
        Usage:

            tasks_obj=Tasks()

            tasks_obj.render_form(request_data, user_id, project_id)

        Args:
            request_data (FormRender(Pydantic Model)): The details to render form.

            user_id (str): The id of the user who is saving this form.

            project_id (str): The id of the project that this form belongs to.

        Returns:
            response (Object): Response containing step metadata and submitted_data
        """
        try:
            common_utils = CommonUtils()
            api_url = f"{self.form_render_url}{APIEndPoints.api_form}"
            cookies = common_utils.generate_cookie_from_user_project(user_id=user_id, project_id=project_id)
            return common_utils.hit_external_service(api_url=api_url, payload=request_data.dict(), method="post", request_cookies=cookies)
        except Exception as e:
            print(e)
            raise

    def save_form(self, request_data: SaveForm, user_id: str, project_id: str):

        """
        Info:
            The save_form function is used to save the form data.
        Usage:

            tasks_obj=Tasks()

            tasks_obj.save_form(request_data, user_id, project_id)

        Args:
            request_data (SaveForm(Pydantic Model)): The SaveForm object containing the form data.

            user_id (str): The id of the user who is saving this form.

            project_id (str): The id of the project that this form belongs to.

        Returns:
            response (Object): Returns success response as "Form Saved Successfully"
        """
        try:
            common_utils = CommonUtils()
            api_url = f"{self.form_render_url}{APIEndPoints.api_form}?save=True"
            cookies = common_utils.generate_cookie_from_user_project(user_id=user_id, project_id=project_id)
            return common_utils.hit_external_service(api_url=api_url, payload=request_data.dict(), method="post", request_cookies=cookies)
        except Exception as e:
            print(e)
            raise

    def fetch_task_details(self, request_data: TaskDetails, user_id: str, project_id: str):

        """
        Info:
            The fetch_task_details function is used to fetch the details of a task.
        Usage:

            tasks_obj=Tasks()

            tasks_obj.fetch_task_details(request_data, user_id, project_id)

        Args:
            request_data (TaskDetails(Pydantic Model)): The TaskDetails object containing the task_id and project_id.

            user_id (str): The id of the user who is saving this form.

            project_id (str): The id of the project that this form belongs to.

        Returns:
            response (Object): Returns task meta details.
        """
        try:
            common_utils = CommonUtils()
            api_url = f"{self.workflow_task_url}{APIEndPoints.api_fetch}"
            cookies = common_utils.generate_cookie_from_user_project(user_id=user_id, project_id=project_id)
            return common_utils.hit_external_service(api_url=api_url, payload=request_data.dict(), method="post",
                                                     request_cookies=cookies)
        except Exception as e:
            print(e)
            raise

    def list_all_tasks(self, request_data: TaskListSchema, user_id: str, project_id: str):

        """
        Info:
            The list_all_tasks function is used to list all the tasks in a project.
        Usage:

            tasks_obj=Tasks()

            tasks_obj.list_all_tasks(request_data, user_id, project_id)

        Args:
            request_data (TaskListSchema(Pydantic Model)): The TaskListSchema schema containing details to list all tasks.

            user_id (str): The id of the user who is saving this form.

            project_id (str): The id of the project that this form belongs to.

        Returns:
            response (Object): Returns all tasks details.
        """
        try:
            common_utils = CommonUtils()
            api_url = f"{self.workflow_task_url}{APIEndPoints.api_list}"
            cookies = common_utils.generate_cookie_from_user_project(user_id=user_id, project_id=project_id)
            return common_utils.hit_external_service(api_url=api_url, payload=request_data.dict(), method="post",
                                                     request_cookies=cookies)
        except Exception as e:
            print(e)
            raise

