import os
import pathlib
import shutil
import sys
from typing import Any, Optional

from dotenv import load_dotenv
from pydantic import BaseSettings, Field, model_validator

load_dotenv()

PROJECT_NAME = "Task Builder SDK"


class _Service(BaseSettings):
    MODULE_NAME: str = Field(default="Task Builder SDK")
    LOG_LEVEL: str = "INFO"
    ENABLE_FILE_LOG: Optional[Any] = False
    ENABLE_CONSOLE_LOG: Optional[Any] = True

    @model_validator
    def validate_values(cls, values):
        values["LOG_LEVEL"] = values["LOG_LEVEL"] or "INFO"
        print(f"Logging Level set to: {values['LOG_LEVEL']}")
        return values


class _PathToServices(BaseSettings):
    FORM_MT: str = Field(None, env="FORM_MT")
    WORKFLOW_MT: str = Field(None, env="WORKFLOW_MT")

    @model_validator
    def validate_values(cls, values):
        if not bool(values["FORM_MT"]):
            print("FORM MT Environment variable not set")
            sys.exit(1)
        if not bool(values["WORKFLOW_MT"]):
            print("Error, environment variable WORKFLOW MANAGEMENT PROXY not set")
            sys.exit(1)
        return values


class _DatabaseConstants(BaseSettings):
    TEMPLATES_DB: str = Field(default="templates_db")
    METADATA_DB: str = Field(default="metadata_db")
    ILENS_ASSISTANT_DB: str = Field(default="ilens_assistant")
    ILENS_MES_DB: str = Field(default="ilens_mes_db")
    ILENS_ASSET_MODEL: str = Field(default="ilens_asset_model")


class _Databases(BaseSettings):
    MONGO_URI: Optional[str]
    POSTGRES_URI: Optional[str]
    USE_POSTGRES: Optional[bool]
    MAINTENANCE_URI: Optional[str] = "maintenance_logbook_qa"
    MAINTENANCE_DB_URI: Optional[str]
    KAIROS_URI: Optional[str]
    REDIS_URI: Optional[str]
    REDIS_LOGIN_DB: Optional[int] = 9
    REDIS_USER_ROLE_DB: Optional[int] = 21
    REDIS_HIERARCHY_DB: Optional[int] = 7
    REDIS_PROJECT_DB: Optional[int] = 18
    PG_ASSISTANT_DB: Optional[str] = "ilens_assistant"
    PG_SCHEMA: Optional[str] = "public"
    PG_REMOVE_PREFIX: bool = False

    @model_validator
    def validate_values(cls, values):
        if not values["MONGO_URI"]:
            print("Error, environment variable MONGO_URI not set")
            sys.exit(1)
        if not values["MAINTENANCE_URI"]:
            print("Environment variable MAINTENANCE_URI not set, proceeding without Postgres Support")
            sys.exit(1)
        if not values["KAIROS_URI"]:
            print("KAIROS_URL env variables missing, continuing without Kafka/Kairos support")
        if not values["POSTGRES_URI"]:
            print("Environment variable POSTGRES_URI not set, proceeding without Postgres Support")
            sys.exit(1)
        values["MAINTENANCE_DB_URI"] = f"{values['POSTGRES_URI']}/{values['MAINTENANCE_URI']}"
        return values


class _KeyPath(BaseSettings):
    KEYS_PATH: Optional[pathlib.Path] = Field("data/keys")
    PUBLIC: Optional[pathlib.Path]
    PRIVATE: Optional[pathlib.Path]

    @model_validator
    def assign_values(cls, values):
        if not os.path.isfile(os.path.join(values.get("KEYS_PATH"), "public")) or not os.path.isfile(
            os.path.join(values.get("KEYS_PATH"), "private")
        ):
            if not os.path.exists(values.get("KEYS_PATH")):
                os.makedirs(values.get("KEYS_PATH"))
            shutil.copy(os.path.join("assets", "keys", "public"), os.path.join(values.get("KEYS_PATH"), "public"))
            shutil.copy(os.path.join("assets", "keys", "private"), os.path.join(values.get("KEYS_PATH"), "private"))
        values["PUBLIC"] = os.path.join(values.get("KEYS_PATH"), "public")
        values["PRIVATE"] = os.path.join(values.get("KEYS_PATH"), "private")
        return values


Service = _Service()
PathToServices = _PathToServices()
DatabaseConstants = _DatabaseConstants()
DBConf = _Databases()
KeyPath = _KeyPath()

__all__ = [
    "PROJECT_NAME",
    "Service",
    "PathToServices",
    "DatabaseConstants",
    "DBConf",
    "KeyPath",
]
