import json
import time
from datetime import datetime, timedelta
from typing import Any

import httpx
import pandas as pd
import pytz
import shortuuid

from tb_sdk.config import PathToServices
from tb_sdk.connectors.constants import APIEndPoints

from tb_sdk.connectors.schemas.logbook_schema import FetchLogbookInfo, LogbookListRequest
from tb_sdk.connectors.utils.common_utils import CommonUtils


class Logbooks:
    def __init__(self):
        self.workflow_task_url = f"{PathToServices.WORKFLOW_MT}{APIEndPoints.logbook_proxy}"

    def fetch_logbook_details(self, request_data: FetchLogbookInfo, user_id: str, project_id: str):

        """
        Info:
            The fetch_logbook_details function is used to fetch the logbook details for a given workflow task.
        Usage:

            logbooks_obj = Logbooks()

            logbooks_obj.fetch_logbook_details(request_data, user_id, project_id)

        Args:
            request_data (FetchLogbookInfo(Pydantic Model)): The FetchLogbookInfo object containing the logbook_id and project_id.

            user_id (str): The id of the user who is saving this form.

            project_id (str): The id of the project that this form belongs to.

        Returns:
            response (Object): Response containing logbook details.
        """
        try:
            common_utils = CommonUtils()
            api_url = f"{self.workflow_task_url}{APIEndPoints.api_get}"
            cookies = common_utils.generate_cookie_from_user_project(user_id=user_id, project_id=project_id)
            return common_utils.hit_external_service(api_url=api_url, payload=request_data.dict(), method="post", request_cookies=cookies)
        except Exception as e:
            print(e)
            raise

    def list_logbooks(self, request_data: LogbookListRequest, user_id: str, project_id: str):

        """
        Info:
            The list_logbooks function is used to list all the logbooks for a given project.
        Usage:

            logbooks_obj = Logbooks()

            logbooks_obj.list_logbooks(request_data, user_id, project_id)

        Args:
            request_data (LogbookListRequest(Pydantic Model)): The request data containing the parameters required to list logbooks.

            user_id (str): The id of the user who is saving this form.

            project_id (str): The id of the project that this form belongs to.

        Returns:
            response (Object): Response containing list of logbook details.
        """
        try:
            common_utils = CommonUtils()
            api_url = f"{PathToServices.WORKFLOW_MT}{APIEndPoints.api_list_logbook_data}"
            cookies = common_utils.generate_cookie_from_user_project(user_id=user_id, project_id=project_id)
            return common_utils.hit_external_service(api_url=api_url, payload=request_data.dict(), method="post", request_cookies=cookies)
        except Exception as e:
            print(e)
            raise
