import os

from fastapi import Request
from sqlalchemy import create_engine
from sqlalchemy.engine.reflection import Inspector
from sqlalchemy.ext.declarative import declarative_base
from sqlalchemy.orm import sessionmaker
from sqlalchemy.schema import CreateSchema
from sqlalchemy_utils import create_database, database_exists

from tb_sdk.config import DBConf
from tb_sdk.connectors.db.redis_connections import project_details_db
from tb_sdk.connectors.utils.redis_db_name_util import get_db_name

Base = declarative_base()


def get_assistant_db(request_data: Request):
    if not DBConf.USE_POSTGRES:
        return
    project_id = request_data.cookies.get("projectId", request_data.cookies.get("project_id"))
    db = (
        get_db_name(redis_client=project_details_db, project_id=project_id, database=DBConf.PG_ASSISTANT_DB)
        if not DBConf.PG_REMOVE_PREFIX
        else DBConf.PG_ASSISTANT_DB
    )
    uri = f"{DBConf.POSTGRES_URI}/{db}"
    engine = create_engine(
        f"{uri}",
        pool_size=int(os.getenv("PG_POOL_SIZE")),
        max_overflow=int(os.getenv("PG_MAX_OVERFLOW")),
    )
    if not database_exists(engine.url):
        create_database(engine.url)
    inspector = Inspector.from_engine(engine)
    if DBConf.PG_SCHEMA not in inspector.get_schema_names():
        engine.execute(CreateSchema(DBConf.PG_SCHEMA, quote=True))
    session_local = sessionmaker(autocommit=False, autoflush=False, bind=engine)

    db = session_local()
    try:
        yield db
    finally:
        db.close()
