import json
import time
from datetime import datetime, timedelta
from typing import Any

import httpx
import pandas as pd
import pytz
import shortuuid

from tb_sdk.config import Service, PathToServices
from tb_sdk.connectors.constants import APIEndPoints

from tb_sdk.connectors.schemas.workflow_schema import FetchWorkFlowResponse, ListWorkflowRequest
from tb_sdk.connectors.utils.common_utils import CommonUtils


class Workflows:
    def __init__(self):
        self.form_render_url = f"{PathToServices.FORM_MT}{APIEndPoints.render_proxy}"
        self.workflow_task_url = f"{PathToServices.WORKFLOW_MT}{APIEndPoints.workflow_proxy}"

    def fetch_workflow_details(self, request_data: FetchWorkFlowResponse, user_id: str, project_id: str):
        """
        Info:
            The fetch_workflow_details function is used to fetch the workflow details from the Workflow Task API.
        Usage:

            workflow_obj = Workflows()

            workflow_obj.fetch_workflow_details(request_data, user_id, project_id)

        Args:
            request_data (FetchWorkFlowResponse(Pydantic Model)): The request data containing the parameters required to fetch workflow details.

            user_id (str): The id of the user who is saving this form.

            project_id (str): The id of the project that this form belongs to.

        Returns:
            response (Object): Response containing workflow details.
        """
        try:
            common_utils = CommonUtils()
            api_url = f"{self.workflow_task_url}{APIEndPoints.api_get}"
            cookies = common_utils.generate_cookie_from_user_project(user_id=user_id, project_id=project_id)
            return common_utils.hit_external_service(api_url=api_url, payload=request_data.dict(), method="post", request_cookies=cookies)
        except Exception as e:
            print(e)
            raise

    def list_workflows(self, request_data: ListWorkflowRequest, user_id: str, project_id: str):

        """
        Info:
            The list_workflows function is used to list all the workflows for a given project.
        Usage:

            workflow_obj = Workflows()

            workflow_obj.list_workflows(request_data, user_id, project_id)

        Args:
            request_data (ListWorkflowRequest(Pydantic Model)): The request object containing the parameters required to list workflows.

            user_id (str): The id of the user who is saving this form.

            project_id (str): The id of the project that this form belongs to.

        Returns:
            response (Object): Response containing list of workflow details.
        """
        try:
            common_utils = CommonUtils()
            api_url = f"{self.workflow_task_url}{APIEndPoints.api_list_workflow_data}"
            cookies = common_utils.generate_cookie_from_user_project(user_id=user_id, project_id=project_id)
            return common_utils.hit_external_service(api_url=api_url, payload=request_data.dict(), method="post", request_cookies=cookies)
        except Exception as e:
            print(e)
            raise
