import json
import time
from datetime import datetime, timedelta
from typing import Any

import httpx
import pandas as pd
import pytz
import shortuuid

from tb_sdk.config import Service, PathToServices
from tb_sdk.connectors.constants import APIEndPoints

from tb_sdk.connectors.schemas.step_schema import ListStepRequest, FetchStepSchema
from tb_sdk.connectors.utils.common_utils import CommonUtils


class Steps:
    def __init__(self):
        self.workflow_task_url = f"{PathToServices.WORKFLOW_MT}{APIEndPoints.step_proxy}"

    def fetch_step_details(self, request_data: FetchStepSchema, user_id: str, project_id: str):
        """
        Info:
            The fetch_step_details function is used to fetch the details of a step.
        Usage:

            step_obj = Steps()

            step_obj.fetch_step_details(request_data, user_id, project_id)

        Args:
            request_data (FetchStepSchema(Pydantic Model)): The request data containing the parameters required to fetch step details.

            user_id (str): The id of the user who is saving this form.

            project_id (str): The id of the project that this form belongs to.

        Returns:
            response (Object): Response containing step details.
        """
        try:
            common_utils = CommonUtils()
            api_url = f"{self.workflow_task_url}{APIEndPoints.api_fetch_step_data}"
            cookies = common_utils.generate_cookie_from_user_project(user_id=user_id, project_id=project_id)
            return common_utils.hit_external_service(api_url=api_url, payload=request_data.dict(), method="post", request_cookies=cookies)
        except Exception as e:
            print(e)
            raise

    def list_steps(self, request_data: ListStepRequest, user_id: str, project_id: str):

        """
        Info:
            The list_steps function is used to list all the steps in a workflow.
        Usage:

            step_obj = Steps()

            step_obj.list_steps(request_data, user_id, project_id)

        Args:
            request_data (ListStepRequest(Pydantic Model)): The request object containing the parameters required to list steps.

            user_id (str): The id of the user who is saving this form.

            project_id (str): The id of the project that this form belongs to.

        Returns:
            response (Object): Response containing list of step details.
        """
        try:
            common_utils = CommonUtils()
            api_url = f"{PathToServices.WORKFLOW_MT}{APIEndPoints.api_list_step_data}"
            cookies = common_utils.generate_cookie_from_user_project(user_id=user_id, project_id=project_id)
            return common_utils.hit_external_service(api_url=api_url, payload=request_data.dict(), method="post", request_cookies=cookies)
        except Exception as e:
            print(e)
            raise
