from typing import Any, Dict, List, Optional

from tb_sdk.connectors.constants import UserCollectionKeys
from tb_sdk.connectors.constants.app_constants import CollectionNames, DatabaseNames
from tb_sdk.connectors.constants import UserKeys
from tb_sdk.connectors.db.mongo.ilens_configuration.collections.user_project import UserProject
from tb_sdk.connectors.db.mongo.schema import MongoBaseSchema
from tb_sdk.connectors.utils.mongo_util import MongoCollectionBaseClass


class UserSchema(MongoBaseSchema):
    name: Optional[str]
    project_id: Optional[str]
    username: Optional[str]
    password: Optional[str]
    email: Optional[str]
    phonenumber: Optional[Any]
    userrole: Optional[List[str]]
    user_type: Optional[str]
    user_id: Optional[str]
    AccessLevel: Optional[Dict]
    user_access_select_all: Optional[bool]
    access_group_ids: Optional[List[str]]
    client_id: Optional[str]
    created_by: Optional[str]
    hmi: Optional[Dict]
    encryption_salt: Optional[Dict]
    product_encrypted: Optional[bool]
    email_preferences: Optional[Dict]
    language: Optional[str]
    passwordReset: Optional[Dict]
    failed_attempts: Optional[int]
    is_user_locked: Optional[bool]
    last_failed_attempt: Optional[str]
    profileImage_name: Optional[str]
    profileImage_url: Optional[str]
    date_format: Optional[str]
    date_time_format: Optional[str]
    time_format: Optional[str]
    tz: Optional[str]
    app_url: Optional[str]
    landing_page: Optional[str]
    ilens_encrypted: Optional[bool]


class User(MongoCollectionBaseClass):
    def __init__(self, mongo_client):
        super().__init__(mongo_client, database=DatabaseNames.ilens_configuration, collection=CollectionNames.user)
        self.user_project_mongo = UserProject(mongo_client=mongo_client)

    @property
    def key_username(self):
        return UserCollectionKeys.KEY_USERNAME

    @property
    def key_user_id(self):
        return UserCollectionKeys.KEY_USER_ID

    @property
    def key_language(self):
        return UserCollectionKeys.KEY_LANGUAGE

    @property
    def key_name(self):
        return UserCollectionKeys.KEY_NAME

    @property
    def key_project_id(self):
        return UserCollectionKeys.KEY_PROJECT_ID

    @property
    def key_userrole(self):
        return UserCollectionKeys.KEY_USER_ROLE

    def get_all_users(self, filter_dict=None, sort=None, skip=0, limit=None, **query):
        if users := self.find(filter_dict=filter_dict, sort=sort, skip=skip, limit=limit, query=query):
            return list(users)
        return []

    def find_user(self, user_id):
        if user := self.find_one(query={self.key_user_id: user_id}):
            return dict(user)
        else:
            return {}

    def find_user_by_param(self, **query):
        return UserSchema(**user) if (user := self.find_one(query)) else user

    def find_access_site(self, user_id):
        if access_site := self.distinct(query_key="AccessLevel.sites.parent_id", filter_json={"user_id": user_id}):
            return list(access_site)
        return []

    def find_by_query_key(self, query_key, user_id):
        if access_site := self.distinct(query_key=query_key, filter_json={self.key_user_id: user_id}):
            return access_site
        return []

    def update_one_user(self, user_id, data):
        query = {self.key_user_id: user_id}
        return self.update_one(query=query, data=data)

    def delete_one_user(self, user_id):
        return self.delete_one(query={self.key_user_id: user_id})

    def users_by_project_and_site(self, user_id, project_id, site_ids):
        query = {"$or": [{self.key_user_id: user_id}, {"AccessLevel.sites.parent_id": {"$in": site_ids}}]}
        if project_id is not None:
            query["$or"].append({self.key_project_id: project_id})
        response = self.find(query)
        return response or []

    def find_user_time_zone(self, user_id):
        search_json = {"_id": 0, "tz": 1, "date_format": 1, "time_format": 1, "date_time_format": 1}
        return self.find_one(filter_dict=search_json, query={self.key_user_id: user_id})

    def distinct_user(self, query_key, filter_json):
        query = {self.key_user_id: filter_json}
        return self.distinct(query_key=query_key, filter_json=query)

    def users_list_by_aggregate(self, query: list):
        return self.aggregate(pipelines=query)

    def find_user_filter(self, project_id, user_id=None, username=None, filter_dict=None):
        query = {}
        if user_id:
            query[self.key_user_id] = user_id
        if username:
            query[self.key_username] = username
        user = self.find_one(query=query, filter_dict=filter_dict)
        if user:
            if project_id != user["project_id"]:
                user_project_record = self.user_project_mongo.fetch_user_project(
                    user_id=user.get(self.key_user_id), project_id=project_id
                )
                if bool(user_project_record):
                    for item in UserKeys.project_specific_keys:
                        user[item] = user_project_record.get(item, None)
            return UserSchema(**user)
        return user

    def find_user_by_project_id(self, user_id, project_id):
        if user := self.find_one(query={self.key_user_id: user_id, self.key_project_id: project_id}):
            return dict(user)
        else:
            return user

    def find_user_data_with_roles(self, roles, project_id):
        query = {self.key_userrole: {"$in": roles}, self.key_project_id: project_id}
        return response if (response := list(self.find(query))) else []

    def find_user_role_for_user_id(self, user_id, project_id):
        query = {self.key_user_id: user_id, self.key_project_id: project_id}
        filter_dict = {"userrole": 1, "_id": 0}
        return self.find_one(query=query, filter_dict=filter_dict)
