"""
Author: Owaiz Mustafa Khan
Email: owaiz.mustafakhan@rockwellautomation.com
"""


import platform

from fastapi import status, HTTPException
from psycopg2 import NotSupportedError

from scripts.constants.db import MongoConstants
from scripts.schemas.postgres_schema import GetTablesInfoResponse, AddIndex, \
    UpdateIndexResponse, UpdateIndex, DeleteIndex, DeleteIndexResponse, AddIndexResponse
from scripts.schemas.util_schema import DetectOS
from scripts.utils.common.db.mongo import find_all, get_collection, find_index, delete_index, add_index, update_index
from scripts.utils.common.db.postgres import create_index_from_data, recreate_index, drop_index_from_data


class CommonUtils:
    @staticmethod
    def detect_os() -> DetectOS:
        """
        Helper Function To Detect OS
        :return: **DetectOS**
        """
        result = DetectOS(
            os_name = platform.system(),
            os_version=platform.version(),
            os_release= platform.release(),
            message="Unknown or unsupported OS."
        )

        print(f"Operating System: {result.os_name}")
        print(f"OS Release: {result.os_release}")
        print(f"OS Version: {result.os_version}")

        if result.os_name == "Windows":
            result.message = "This is a Windows system."
        elif result.os_name == "Linux":
            result.message = "This is a Linux system."
        elif result.os_name == "Darwin":
            result.message = "This is macOS."
        print(result.message)
        return result

    if __name__ == "__main__":
        print(detect_os())

    def get_tables_info(self):
        """
        This function is used to get all the records from the collection 'postgres_default_schema_info'
        :return: **GetTablesInfoResponse**: all the records of the collection or False if any error occurs
        """
        try:
            collection = get_collection(
                MongoConstants.collection_postgres_default_schema_info,
                MongoConstants.db_default_info
            )
            records = find_all(collection)

            records = {'data': records}

            return GetTablesInfoResponse(**records)
        except Exception as e:
            print(f'Exception occurred: {e}')
            return False

    def update_index_from_db(self) -> dict:
        """
        This function applies all the indexes to the postgres tables on the basis of data in Mongo
        :return: dict response {'status': 'SUCCESS', 'message': 'All the indexes from the db are applied to postgres'}
        """
        try:
            collection = get_collection(
                MongoConstants.collection_postgres_default_schema_info,
                MongoConstants.db_default_info
            )
            records = find_index(collection)

            for record in records:
                record = UpdateIndex(**dict(record))
                create_index_from_data(record.model_dump())

            return {
                'status': 'SUCCESS',
                'message': 'All the indexes from the db are applied to postgres'
            }

        except Exception as e:
            print(f'Exception occurred: {e}')
            raise HTTPException(
                status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
                detail=f'''There was some technical error at our end we'll resolve it quickly. Thank you for your patience.'''
            )


    def add_index(self, payload: AddIndex):
        """
        This is a dynamic functions that Adds Index if not present or Updates Index if present subsequently updating the metadata in Mongo

        AddIndex:
            database_name: Name of database you want the index to be added or updated on

            schema_name: Name of schema you want the index to be added or updated on

            table_name: Name of table you want the index to be added or updated on

            index: Class: PostgresTableIndex
                name: Name of the Index

                columns: Array of columns you want to include in index

                type: The Access Method of Index

                unique: [true/false]

        :param payload: Class: AddIndex(Information of AddIndex is Mentioned Above)
        :return: **AddIndexResponse** or **UpdateIndexResponse** the newly added or updated index info record of Mongo
        """
        try:
            collection = get_collection(
                MongoConstants.collection_postgres_default_schema_info,
                MongoConstants.db_default_info
            )
            exists = True
            records = find_index(
                collection,
                index_name=payload.index.name,
                table_name=payload.table_name,
                schema_name=payload.schema_name,
                database_name=payload.database_name
            )
            if not len(records):
                exists = False
                # Till here index of same name doesn't exist
                # Now will check for columns
                records = find_index(
                    collection,
                    database_name=payload.database_name,
                    schema_name=payload.schema_name,
                    table_name=payload.table_name
                )
                for record in records:
                    record = UpdateIndex(**record)
                    if (
                            (payload.index.columns == record.index.columns)
                            and
                            (payload.index.type == record.index.type)
                    ):
                        records = [record.model_dump()]
                        exists = True
                        break

            if exists:
                for record in records:
                    index_data = dict(record)
                    record = UpdateIndexResponse(**index_data)
                    index_data.update({'new_index': payload.index.model_dump()})
                    recreate_index(index_data)
                    if not update_index(collection, index_data):
                        return HTTPException(
                            status_code=status.HTTP_417_EXPECTATION_FAILED,
                            detail="""Unable to update index metadata in database"""
                        )
                    return record

            create_index_from_data(payload.model_dump())
            if not add_index(collection, payload.model_dump()):
                return HTTPException(
                    status_code=status.HTTP_417_EXPECTATION_FAILED,
                    detail="""Unable to add metadata to the database"""
                )

            return AddIndexResponse(**payload.model_dump())

        except NotSupportedError as e:
            if 'access method "hash" does not support multicolumn indexes' in str(e):
                raise HTTPException(
                    status_code=status.HTTP_406_NOT_ACCEPTABLE,
                    detail="HASH Doesn't Support Multiple Columns"
                )

        except Exception as e:
            print(f'Exception occurred: {e}')
            raise HTTPException(
                status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
                detail=f'''There was some technical error at our end we'll resolve it quickly. Thank you for your patience.'''
            )


    def delete_index(self, payload: DeleteIndex) -> None | HTTPException | DeleteIndexResponse:
        """
        This functions Deletes the index if it exists subsequently deleting the metadata in Mongo

        DeleteIndex:
            name: Name of the Index to delete

        :param payload: Class: DeleteIndex(Information of DeleteIndex is Mentioned Above)
        :return: **DeleteIndexResponse** the deleted index info record of Mongo or **HTTPException**
        """
        try:
            collection = get_collection(
                MongoConstants.collection_postgres_default_schema_info,
                MongoConstants.db_default_info
            )
            records = find_index(
                collection,
                index_name=payload.name
            )

            if not len(records):
                return HTTPException(
                    status_code=status.HTTP_404_NOT_FOUND,
                    detail=f'Failed To Delete.... No Index Like "{payload.name}" Exists'
                )

            for record in records:
                index_data = dict(record)
                record = DeleteIndexResponse(**record)
                drop_index_from_data(index_data)
                if not delete_index(collection, index_data):
                    return HTTPException(
                        status_code=status.HTTP_501_NOT_IMPLEMENTED,
                        detail="""Unable to delete metadata for db"""
                    )
                return record

        except Exception as e:
            print(f'Exception occurred: {e}')
            raise HTTPException(
                status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
                detail=f'''There was some technical error at our end we'll resolve it quickly. Thank you for your patience.'''
            )
