import json
import csv
import logging
from pymongo import MongoClient


def setup_logging():
    logging.basicConfig(
        level=logging.INFO,
        format='%(asctime)s | %(levelname)s | %(message)s',
        handlers=[
            logging.StreamHandler(),
        ]
    )


def connect_to_mongo(uri):
    try:
        logging.info("Connecting to MongoDB...")
        client = MongoClient(uri)
        client.admin.command('ping')
        logging.info("Connected to MongoDB.")
        return client
    except Exception as e:
        logging.error(f"Connection failed: {e}")
        exit(1)


def get_user_defined_indexes(client):
    result = []
    json_output = {}

    try:
        databases = client.list_database_names()
    except Exception as e:
        logging.error(f"Failed to list databases: {e}")
        exit(1)

    for db_name in databases:
        if db_name in ("admin", "local", "config"):
            continue

        db = client[db_name]
        json_output[db_name] = {}

        try:
            collections = db.list_collection_names()
        except Exception as e:
            logging.error(f"Failed to list collections in '{db_name}': {e}")
            continue

        for collection_name in collections:
            collection = db[collection_name]
            try:
                indexes = collection.index_information()
                user_indexes = []

                for index_name, index_data in indexes.items():
                    if index_name == "_id_":
                        continue

                    index_info = {
                        "database": db_name,
                        "collection": collection_name,
                        "name": index_name,
                        "key": str(index_data.get("key")),
                        "unique": index_data.get("unique", False),
                        "sparse": index_data.get("sparse", False),
                        "expireAfterSeconds": index_data.get("expireAfterSeconds"),
                        "partialFilterExpression": str(index_data.get("partialFilterExpression")),
                        "collation": str(index_data.get("collation"))
                    }

                    result.append(index_info)
                    user_indexes.append(index_info)

                json_output[db_name][collection_name] = user_indexes

            except Exception as e:
                logging.error(f"Failed to get indexes for {collection_name} in {db_name}: {e}")

    return result, json_output


def save_to_json(data, filepath="mongo_indexes.json"):
    try:
        with open(filepath, "w", encoding="utf-8") as f:
            json.dump(data, f, indent=4)
        logging.info(f"Index data saved to JSON: {filepath}")
    except Exception as e:
        logging.error(f"Failed to save JSON: {e}")


def save_to_csv(data, filepath="mongo_indexes.csv"):
    try:
        with open(filepath, mode="w", newline="", encoding="utf-8") as f:
            fieldnames = [
                "database", "collection", "name", "key", "unique",
                "sparse", "expireAfterSeconds", "partialFilterExpression", "collation"
            ]
            writer = csv.DictWriter(f, fieldnames=fieldnames)
            writer.writeheader()
            writer.writerows(data)
        logging.info(f"Index data saved to CSV: {filepath}")
    except Exception as e:
        logging.error(f"Failed to save CSV: {e}")


def main():
    setup_logging()

#    uri = "mongodb://dev-mongo-readwrite:dev-mongo-readwrite%40123@192.168.0.220:31589/?directConnection=true"
    uri="mongodb://iLens:iLens%231234@192.168.0.207:3599/?authMechanism=DEFAULT&directConnection=true"
    client = connect_to_mongo(uri)
    index_list, json_structure = get_user_defined_indexes(client)

    save_to_json(json_structure)
    save_to_csv(index_list)

    logging.info("Index extraction completed.")


if __name__ == "__main__":
    main()