"""
Author: Owaiz Mustafa Khan
Email: owaiz.mustafakhan@rockwellautomation.com
"""



import pymongo
from pymongo.errors import OperationFailure

from scripts.schemas.mongo_schema import DeleteIndex, UpdateIndexFromDB
from scripts.utils.common import get_collection


def get_data(data: str):
    if data == 'ASC':
        return pymongo.ASCENDING
    elif data == 'DESC':
        return pymongo.DESCENDING
    elif data == 'GEO2D':
        return pymongo.GEO2D
    elif data == 'GEOSPHERE':
        return pymongo.GEOSPHERE
    elif data == 'HASHED':
        return pymongo.HASHED
    elif data == 'TEXT':
        return pymongo.TEXT

def get_collection_info(index: dict):
    collection_name = index.get('collection', index.get('collection_name'))
    db_name = index.get('db', index.get('db_name'))
    return collection_name, db_name

def get_index_info(index: dict) -> list[dict]:
    collection_name, db_name = get_collection_info(index)
    collection = get_collection(collection_name, db_name)

    # Get all index from mongo
    all_index = collection.list_indexes()

    # Format to readable data
    index_infos = [
        {"name": idx["name"], "fields": list(idx["key"].keys())}
        for idx in all_index
    ]

    return index_infos

def index_exists(index: dict, by_name: bool = False) -> bool:
    index_infos = get_index_info(index)

    if by_name:
        for i in index_infos:
            if index.get('name') == i.get('name'):
                return True
        return False

    # Compare if index already exists
    for index_info in index_infos:
        if index.get('fields') == index_info.get('fields'):
            return True
    return False

def make_keys(indexes: list[dict]) -> list[tuple] | str:
    result = list()
    keys = indexes[0].get('keys')
    if indexes[0].get('type') == 'simple':
        return keys
    for key in keys:
        result.append((
            key[0],
            get_data(key[1])
        ))
    return result

def add_new_index(index_info):
    try:
        collection_name, db_name = get_collection_info(index_info)
        indexes = index_info.get('indexes')
        additional_properties = indexes[0].get('additional_properties')
        fields = make_keys(indexes)
        collection = get_collection(collection_name, db_name)

        if not additional_properties:
            collection.create_index(keys=fields)
            return

        collection.create_index(
            keys = fields,
            unique=additional_properties.get('unique', False),
            sparse=additional_properties.get('sparse', False),
            hidden=additional_properties.get('hidden', False),
            background=additional_properties.get('background', False)
        )
    except OperationFailure as of:
        if 'Index with name: email_1 already exists with different options' in str(of):
            print('Cannot add index as it already exists')
        return

    except Exception as e:
        print(e)
        return

def update_all_index_from_db(payload: UpdateIndexFromDB):
    try:
        collection = get_collection(payload.collection_name, payload.db_name)

        # Get all index metadata stored in Mongo
        index_infos = list(collection.find({}, {'_id': 0}))

        # Checks that data is present
        if not len(index_infos):
            return {
                'status': 'COMPLETED',
                'message': 'No index metadata found to create indexes.'
            }

        # Iterate through data and add new index
        for index_info in index_infos:
            if index_exists(index_info): # Checking if the index already exists
                continue
            add_new_index(index_info)    # Adding new index
    except Exception as e:
        print(f'Exception occurred during update: {e}')
        exit(0)

def delete_index(payload: DeleteIndex):
    try:
        i_collection = get_collection(payload.collection_name, payload.db_name)

        if not index_exists(payload.model_dump(), by_name=True):
            return

        collection = get_collection(payload.metadata_collection_name, payload.metadata_db_name)

        index_infos = get_index_info(payload.model_dump())

        # Deleting metadata from mongo
        docs = collection.find({"db": payload.db_name, "collection": payload.collection_name})

        for doc in docs:
            indexes = doc.get("indexes", [])
            new_indexes = []

            for index in indexes:
                index_type = index.get("type")
                keys = index.get("keys")

                if isinstance(keys, list):
                    _ = list()
                    for key in keys:
                        _.append(key[0])

                    keys = _

                for index_info in index_infos:
                    if ([keys] if index_type == 'simple' else keys) == index_info.get('fields'):
                        print(f"Deleting entire document: {doc['_id']}")
                        collection.delete_one({"_id": doc["_id"]})
                        break
                else:
                    new_indexes.append(index)

            else:
                # If loop wasn't broken (i.e., not deleted), update or delete based on new index array
                if not new_indexes:
                    collection.delete_one({"_id": doc["_id"]})
                else:
                    collection.update_one(
                        {"_id": doc["_id"]},
                        {"$set": {"indexes": new_indexes}}
                    )
        i_collection.drop_index(payload.name)

    except Exception as e:
        print(f'Some exception occurred while deleting the index: {e}')
        exit(0)


delete_index(DeleteIndex(
    name='lookup_name_-1_lookup_id_-1',
    collection_name='lookup_table',
    metadata_collection_name='test_collection',
    db_name='ilens_configuration',
    metadata_db_name='__test'
))

# delete_index(DeleteIndex(
#     name='id_1',
#     collection_name='design_tag_data',
#     metadata_collection_name='test_collection',
#     db_name='ilens_configuration',
#     metadata_db_name='__test'
# ))


# update_all_index_from_db(UpdateIndexFromDB())