from typing import Dict, List, Optional

from pydantic import BaseModel

from scripts.db.redis_connection import destination_space_db
from scripts.utils.mongo_utils import MongoCollectionBaseClass

from scripts.constants.db_constants import (
    CollectionNames,
    DatabaseNames,
)


class AssetModelMappingSchema(BaseModel):
    """
    This is the Schema for the Mongo DB Collection.
    All datastore and general responses will be following the schema.
    """

    asset_model_id: Optional[str] = ""
    asset_version: Optional[str] = ""
    artifact_id: Optional[str] = ""
    parameter_details: Optional[List] = []
    process_details: Optional[List] = []
    industry_details: Optional[Dict] = {}


class AssetModelMapping(MongoCollectionBaseClass):
    def __init__(self, mongo_client, space_id=None):
        super().__init__(
            mongo_client,
            database=DatabaseNames.catalog,
            collection=CollectionNames.asset_model_mapping,
            space_db=destination_space_db
        )
        self.space_id = space_id

    @property
    def key_asset_model_id(self):
        return "asset_model_id"

    @property
    def key_artifact_id(self):
        return "artifact_id"

    @property
    def key_asset_version(self):
        return "asset_version"

    @property
    def key_space_id(self):
        return "space_id"

    def find_asset_detail_by_id(
        self, space_id, artifact_id=None, asset_id=None, asset_version=None, asset_name=None, filter_dict=None
    ):
        query = {}
        query.update({self.key_space_id: space_id})
        if asset_id:
            query.update({self.key_asset_model_id: asset_id})
        if artifact_id:
            query.update({self.key_artifact_id: artifact_id})
        if asset_version:
            query.update({self.key_asset_version: asset_version})
        if asset_name:
            query.update({"asset_model_name": asset_name})
        asset_list = self.find_one(query=query, filter_dict=filter_dict)
        if asset_list:
            return asset_list
        return {}

    def insert_one_asset_detail(self, data):
        """
        The following function will insert one asset in the
        asset_list collections
        :param self:
        :param data:
        :return:
        """
        insert_data = data
        return self.insert_one(insert_data)
