from typing import Dict, List, Optional

from pydantic import BaseModel

from scripts.db.redis_connection import destination_space_db
from scripts.utils.mongo_utils import MongoCollectionBaseClass

from scripts.constants.db_constants import (
    CollectionNames,
    DatabaseNames,
)


class AssetDetailSchema(BaseModel):
    """
    This is the Schema for the Mongo DB Collection.
    All datastore and general responses will be following the schema.
    """

    asset_model_id: Optional[str] = ""
    allow_editing: Optional[bool] = True
    asset_description: Optional[str] = ""
    asset_version: Optional[str] = ""
    asset_model_type: Optional[str] = ""
    asset_model_icon: Optional[str] = ""
    parameters: Optional[Dict] = {}
    parameters_new: Optional[Dict] = {}
    processes: Optional[list] = []
    device_models: Optional[List] = []
    events: Optional[List] = []
    resources: Optional[Dict] = {}
    others: Optional[Dict] = {}


class AssetModelArtifacts(MongoCollectionBaseClass):
    def __init__(self, mongo_client, space_id=None):
        super().__init__(
            mongo_client, database=DatabaseNames.catalog, collection=CollectionNames.asset_model, space_db=destination_space_db
        )
        self.space_id = space_id

    @property
    def key_asset_model_id(self):
        return "asset_model_id"

    @property
    def key_asset_version(self):
        return "asset_version"

    @property
    def key_space_id(self):
        return "space_id"

    def get_highest_asset_model_version(self, asset_model_name, space_id):
        query = [
            {"$match": {"asset_model_name": asset_model_name, "space_id": space_id}},
            {
                "$group": {
                    "_id": "$asset_model_id",
                    "highestVersion": {"$max": "$asset_version"},
                }
            },
        ]
        res = list(self.aggregate(query))
        if res:
            return res[0].get("highestVersion", "0.0"), res[0].get("_id", "")
        else:
            return "0.0", ""

    def insert_one_asset_detail(self, data):
        """
        The following function will insert one asset in the
        asset_list collections
        :param self:
        :param data:
        :return:
        """
        insert_data = data
        return self.insert_one(insert_data)

    def aggregate_asset_detail(self, filter_list: List):
        if filter_list:
            return self.aggregate(pipelines=filter_list)
