import datetime
import json
import os.path

import pytz
import shortuuid

from scripts.db.mongo import source_mongo_client, destination_mongo_client
from scripts.db.mongo.catalog.aggregations.asset_model_details import AssetDetailAggregate
from scripts.db.mongo.catalog.artifact_meta import ArtifactsMeta
from scripts.db.mongo.catalog.asset_model import AssetModelArtifacts
from scripts.db.mongo.catalog.asset_model_mapping import AssetModelMapping
from scripts.db.mongo.catalog.asset_model_rule_engine import AssetRuleEngineMeta
from scripts.db.mongo.ilens_asset_model.asset_model_details import AssetDetail
from scripts.db.mongo.ilens_asset_model.asset_model_rule_engine import AssetModelRuleEngine
from scripts.db.mongo.ilens_asset_model.industry_category import IndustryCategory
from scripts.db.mongo.ilens_configuration.materials import Materials
from scripts.db.mongo.ilens_configuration.process_conf import ProcessConf
from scripts.db.mongo.ilens_configuration.tag_category import TagCategory
from scripts.db.mongo.ilens_configuration.tag_group import TagGroups
from scripts.db.mongo.ilens_configuration.tags import Tags
from scripts.db.mongo.ilens_configuration.unit import Units
from scripts.db.mongo.ilens_configuration.unit_group import UnitGroups
from scripts.logging import logger
from scripts.utils.common_utils import CommonUtils


class MigrateAssetModel:
    def __init__(self, project_id, space_id):
        """
        The __init__ function is called when the class is instantiated.
        It sets up the instance of the class, and defines all of its attributes.


        :param self: Represent the instance of the class
        :param : Pass the mongo client to the class
        :return: The following:
        """
        self.project_id = project_id
        self.space_id = space_id
        self.asset_detail_aggregate = AssetDetailAggregate()
        self.common_utils = CommonUtils(space_id=self.space_id)
        self.destination_asset_model_mapping_mongo = AssetModelMapping(mongo_client=destination_mongo_client,
                                                                       space_id=self.space_id)
        self.destination_asset_model_rule_engine_mongo = AssetRuleEngineMeta(mongo_client=destination_mongo_client,
                                                                             space_id=self.space_id)
        self.destination_artifact_meta = ArtifactsMeta(mongo_client=destination_mongo_client, space_id=self.space_id)
        self.destination_asset_model = AssetModelArtifacts(mongo_client=destination_mongo_client,
                                                           space_id=self.space_id)
        self.source_process_conf = ProcessConf(mongo_client=source_mongo_client)
        self.source_materials = Materials(mongo_client=source_mongo_client, project_id=self.project_id)
        self.source_units = Units(mongo_client=source_mongo_client, project_id=self.project_id)
        self.source_unit_group = UnitGroups(mongo_client=source_mongo_client, project_id=self.project_id)
        self.source_tags = Tags(mongo_client=source_mongo_client, project_id=self.project_id)
        self.source_tag_group = TagGroups(mongo_client=source_mongo_client, project_id=self.project_id)
        self.source_tag_category = TagCategory(mongo_client=source_mongo_client, project_id=self.project_id)
        self.source_asset_model_details = AssetDetail(mongo_client=source_mongo_client, project_id=self.project_id)
        self.source_industry_category = IndustryCategory(mongo_client=source_mongo_client)
        self.source_asset_model_rule_engine_conn = AssetModelRuleEngine(mongo_client=source_mongo_client,
                                                                        project_id=self.project_id)
        self.mapping_dict = {}

    def bind_unit_data(self, unit_id):
        try:
            if not unit_id:
                return {}
            unit_data = self.source_units.find_one({'id': unit_id})
            if not unit_data:
                return {}
            unit_group_data = self.source_unit_group.find_one({'id': unit_data.get("unit_group_id")})
            if not unit_group_data:
                unit_group_data = {}
            unit_data["unit_group"] = unit_group_data
            return unit_data
        except Exception as e:
            logger.error(str(e))

    def get_tags(self, source_tags_list):
        try:
            tags_list = list(self.source_tags.find({"tag_id": {"$in": source_tags_list}}))
            destination_tag_data = []
            for tag in tags_list:
                tag_group_data = self.source_tag_group.find_one({"id": tag.get("tag_group_id", "")})
                tag_category_data = self.source_tag_category.find_one(
                    {"tag_category_id": tag.get("tag_category_id", "")})
                unit_data = self.bind_unit_data(tag.get("unit"))
                tag["tag_group"] = tag_group_data or {}
                tag["tag_category"] = tag_category_data or {}
                tag["unit_data"] = unit_data
                destination_tag_data.append(tag)
            return destination_tag_data
        except Exception as e:
            logger.error(str(e))

    def get_industry_category_details(self, industry_category_id):
        try:
            industry_category_details = self.source_industry_category.find_one(
                {"industry_category_id": industry_category_id}, {"_id": 0})
            if not industry_category_details:
                return {}
            return industry_category_details
        except Exception as e:
            logger.error(str(e))

    def get_material_data(self, asset_model_id, asset_version):
        try:
            asset_model_version = [asset_model_id + "$" + str(asset_version)]
            material_data = list(self.source_materials.find({"asset_models": {"$in": asset_model_version}}, {"_id": 0}))
            destination_material_data = []
            for material in material_data:
                unit_data = self.bind_unit_data(material.get("uom"))
                material["unit_data"] = unit_data
                destination_material_data.append(material)
            return destination_material_data
        except Exception as e:
            logger.error(str(e))

    def get_process_details(self, process_list):
        try:
            return list(self.source_process_conf.find({"process_id": {"$in": process_list}}, {"_id": 0}))
        except Exception as e:
            logger.error(str(e))

    @staticmethod
    def generate_shortuuid_with_alpha_first() -> str:
        # Generate a short UUID
        short_uuid = shortuuid.uuid()

        # Ensure the first character is alphabetic (a-z)
        while not short_uuid[0].isalpha():
            short_uuid = shortuuid.uuid()

        return short_uuid

    def generate_artifact_meta_json(self,
                                    space_id, name, type_, source_details, source_id, artifact_id=None, image="",
                                    ver="1.0"
                                    ):
        if not artifact_id:
            artifact_id = self.generate_shortuuid_with_alpha_first()
        if not image:
            image = f"{artifact_id}.jpg"
        data = {
            "artifact_id": artifact_id,
            "name": name,
            "artifact_type": type_,
            "ver": ver,
            "image": image,
            "status": "pending",
            "meta": {
                "published_by": source_details.get("published_by", ""),
                "published_on": int(datetime.datetime.now(tz=pytz.timezone("UTC")).timestamp()),
            },
            "source_details": source_details,
            "comments": "",
            "space_id": space_id,
            "source_id": source_id,
        }
        return data

    def get_rule_data(self, asset_model_id, asset_version, asset_model_mapping):
        try:
            space_rules = []
            catalog_rules = self.source_asset_model_rule_engine_conn.find({'asset_model_id': asset_model_id,
                                                                           'asset_version': asset_version,
                                                                           })
            for rule in catalog_rules:
                rule_data = self.source_asset_model_rule_engine_conn.find_one(
                    {"rule_engine_id": rule.get("rule_engine_id")}, {"_id": 0})
                if not rule_data:
                    continue
                rule_data["asset_model_id"] = asset_model_mapping.get("asset_model_id")
                rule_data["asset_version"] = asset_model_mapping.get("asset_version")
                rule_data["space_id"] = self.space_id
                rule_data["artifact_id"] = asset_model_mapping.get("artifact_id")
                rule_data["artifact_ver"] = asset_model_mapping.get("artifact_ver")
                space_rules.append(rule_data)
            if space_rules:
                self.destination_asset_model_rule_engine_mongo.insert_many(data=space_rules)

        except Exception as e:
            logger.error(str(e))

    def insert_asset_model_details(self, asset_model_names_list=[]):
        try:
            if not asset_model_names_list:
                asset_model_names_list = list(self.source_asset_model_details.distinct("asset_model_name"))
            logger.info(f"total_count {len(asset_model_names_list)}")
            for asset_name_index, asset_model_name in enumerate(asset_model_names_list):
                logger.info(
                    f"asset_model_name {asset_model_name} asset_model_name index {asset_name_index}")
                # data = self.destination_artifact_meta.find(query={
                #                                                   'name': asset_model_name
                #                                                   })
                # if list(data):
                #     print(f"Skipping: {asset_model_name}")
                #     continue
                source_asset_model_details = list(
                    self.source_asset_model_details.find({"asset_model_name": asset_model_name}, {"_id": 0}))
                for index, asset_model in enumerate(source_asset_model_details):

                    logger.info(
                        f"asset_model_id {asset_model.get('asset_model_id')} asset_version {asset_model.get('asset_version')} index {index}")
                    if asset_model.get("asset_model_id") == "asset_model_705":
                        print("true")
                    if len(asset_model.get("parameters", [])) > 1000:
                        logger.info(
                            f"skipping asset_model as parameter are more than 1000 associated asset_model_id {asset_model.get('asset_model_id')} asset_version {asset_model.get('asset_version')}")
                        continue
                    source_asset_model_id = asset_model.get("asset_model_id")
                    source_asset_version = asset_model.get("asset_version")
                    parameter_data = self.get_tags(asset_model.get("parameters", []))
                    industry_category_details = self.get_industry_category_details(
                        asset_model.get("industry_category_id", ""))
                    material_data = self.get_material_data(source_asset_model_id, source_asset_version)
                    process_details = self.get_process_details(asset_model.get("processes", []))
                    artifact_ver = self.destination_artifact_meta.get_artifact_latest_version(
                        artifact_name=asset_model.get("asset_model_name"), artifact_type="asset_model"
                    )
                    artifact_meta_data = self.generate_artifact_meta_json(
                        space_id=self.space_id,
                        name=asset_model.get("asset_model_name"),
                        type_="asset_model",
                        source_details={
                            "published_by": "Admin"
                        },
                        source_id=source_asset_model_id,
                        artifact_id=None,
                        ver=artifact_ver,
                        image=asset_model.get('asset_model_icon')
                    )
                    asset_model_mapping = {
                        "asset_model_id": asset_model.get("asset_model_id"),
                        "asset_version": asset_model.get("asset_version"),
                        "artifact_id": artifact_meta_data.get("artifact_id"),
                        "artifact_ver": artifact_ver,
                        "space_id": self.space_id,
                        "parameter_details": parameter_data,
                        "industry_category_details": industry_category_details,
                        "materials": material_data,
                        "process_details": process_details
                    }
                    asset_model["artifact_id"] = artifact_meta_data.get("artifact_id")
                    asset_model["artifact_ver"] = artifact_ver
                    # if asset_model.get("rules"):
                    self.mapping_dict[artifact_meta_data.get("artifact_id")] = \
                        {"id": f"{asset_model.get('asset_model_id')}__{asset_model.get('asset_version')}",
                         "image": asset_model.get('asset_model_icon')
                         }
                    self.get_rule_data(asset_model_id=asset_model.get("asset_model_id"),
                                       asset_version=asset_model.get('asset_version'),
                                       asset_model_mapping=asset_model_mapping)
                    self.destination_asset_model_mapping_mongo.insert_one(asset_model_mapping)
                    self.destination_asset_model.insert_one(asset_model)
                    artifact_meta_data["source_details"]["industry"] = asset_model_mapping.get(
                        "industry_category_details", {}
                    ).get("industry_category_name")
                    self.destination_artifact_meta.insert_one(artifact_meta_data)
            logger.info("insert completed")
        except Exception as e:
            logger.error(str(e))
        file = f'/code/data/catalog/mapping_data_{datetime.datetime.now().strftime("%Y_%m_%d_%H_%m")}.json'
        os.makedirs(file, exist_ok=True)
        with open(file, 'w') as f:
            json.dump(self.mapping_dict, f)
