from typing import Optional

from ut_mongo_util import CollectionBaseClass


class UserCollectionKeys:
    KEY_LANGUAGE = "language"
    KEY_NAME = "name"
    KEY_USER_ID = "user_id"
    KEY_SPACE_ID = "space_id"
    KEY_USERNAME = "username"
    KEY_USER_ROLE = "user_role_name"
    KEY_EMAIL = "email"


class UserRole(CollectionBaseClass):
    def __init__(self, mongo_client, space_id=None):
        super().__init__(mongo_client, database="catalog_meta", collection="user_role")
        self.space_id = space_id
        self.key_user_id = UserCollectionKeys.KEY_USER_ID
        self.key_space_id = UserCollectionKeys.KEY_SPACE_ID

    def update_user_role(self, query, data):
        """
        The following function will update target details in rule_targets collections
        :param self:
        :param data:
        :return:
        """
        return self.update_one(query=query, data=data, upsert=True)

    def find_user(self, user_id):
        user = self.find_one(query={"user_id": user_id})
        if user:
            return dict(user)
        return user

    def find_user_name(self, user_id, space_id: Optional[str]):
        query = {"user_role_id": user_id, "space_id": space_id}
        one_user = self.find_one(filter_dict={"user_role_name": 1, "_id": 0}, query=query)
        if one_user is None:
            return one_user
        return one_user["user_role_name"]

    @staticmethod
    def get_users_list(space_id=None):
        query_json = [
            {
                "$group": {
                    "_id": None,
                    "data": {"$push": {"k": {"$ifNull": ["$user_id", ""]}, "v": {"$ifNull": ["$username", ""]}}},
                }
            },
            {"$replaceRoot": {"newRoot": {"$arrayToObject": "$data"}}},
        ]
        if space_id:
            query_json.insert(0, {"$match": {"space_id": space_id}})
        return query_json

    def users_list_by_aggregate(self, query: list):
        return self.aggregate(pipelines=query)

    def find_user_by_space_id(self, user_id, space_id):
        user = self.find_one(query={self.key_user_id: user_id, self.key_space_id: space_id})
        if user:
            return dict(user)
        return user

    def find_user_role_by_id(self, user_role_id, filter_dict=None):
        return self.find_one(query={"user_role_id": user_role_id}, filter_dict=filter_dict)

    def find_user_role_by_aggregate(self, query):
        if role_by_aggregate := list(self.aggregate(query)):
            return role_by_aggregate
        return []
