import os
import pathlib
import shutil
from typing import Optional, Any
from dotenv import load_dotenv
from pydantic.v1 import Field, root_validator, BaseSettings

load_dotenv()


class _LoggVariables(BaseSettings):
    LOG_LEVEL: str = Field(default="DEBUG")
    ENABLE_FILE_LOG: Optional[Any] = Field(default=False)
    ENABLE_CONSOLE_LOG: Optional[Any] = Field(default=True)
    LOGS_MODULE_PATH: Optional[pathlib.Path] = Field(default="/code/data/default_catalog_meta_logs")


class _Databases(BaseSettings):
    SOURCE_MONGO_URI: Optional[str]
    DESTINATION_MONGO_URI: Optional[str]
    SOURCE_REDIS_URI: Optional[str]
    DESTINATION_REDIS_URI: Optional[str]
    REDIS_SPACE_DB: int = Field(default=18)
    REDIS_USER_ROLE_DB: Optional[int] = Field(default=21)


class _Security(BaseSettings):
    ENCRYPTION_CONSTANTS_FILE_PATH: str = "scripts/config/mongo_encryption_constants.json"
    USER_ENCRYPTION: bool = Field(default=True)


class _KeyPath(BaseSettings):
    KEYS_PATH: Optional[pathlib.Path] = Field(default="data/keys")
    PUBLIC: Optional[pathlib.Path]
    PRIVATE: Optional[pathlib.Path]

    @root_validator(allow_reuse=True)
    def assign_values(cls, values):
        if not os.path.isfile(os.path.join(values.get("KEYS_PATH"), "public")) or not os.path.isfile(
            os.path.join(values.get("KEYS_PATH"), "private")
        ):
            if not os.path.exists(values.get("KEYS_PATH")):
                os.makedirs(values.get("KEYS_PATH"))
            shutil.copy(os.path.join("assets", "keys", "public"), os.path.join(values.get("KEYS_PATH"), "public"))
            shutil.copy(os.path.join("assets", "keys", "private"), os.path.join(values.get("KEYS_PATH"), "private"))
        values["PUBLIC"] = os.path.join(values.get("KEYS_PATH"), "public")
        values["PRIVATE"] = os.path.join(values.get("KEYS_PATH"), "private")
        return values


DBConf = _Databases()
LoggVariables = _LoggVariables()
Security = _Security()
KeyPath = _KeyPath()
__all__ = [
    "DBConf",
    "LoggVariables",
    "Security",
    "KeyPath"
]