from typing import Optional

from pydantic import BaseModel

from scripts.constants.db_constants import DatabaseNames, CollectionNames
from scripts.db.redis_connection import source_space_db
from scripts.utils.mongo_utils import MongoCollectionBaseClass


class IndustryCategorySchema(BaseModel):
    """
    This is the Schema for the Mongo DB Collection.
    All datastore and general responses will be following the schema.
    """

    industry_category_name: str
    description: Optional[str] = ""
    industry_category_id: Optional[str] = ""
    upload_icon: Optional[str] = ""
    is_deleted: Optional[str] = ""


class IndustryCategory(MongoCollectionBaseClass):
    def __init__(self, mongo_client):
        super().__init__(
            mongo_client, database=DatabaseNames.ilens_asset_model, collection=CollectionNames.industry_category, space_db=source_space_db
        )

    def find_one_industry_category(self, filter_dict=None, **query) -> IndustryCategorySchema:
        """
        The following function will give one industry_category for a given set of
        search parameters as keyword arguments
        :param filter_dict:
        :param query:
        :return:
        """
        industry = self.find_one(filter_dict=filter_dict, query=query)
        if industry:
            return IndustryCategorySchema(**industry)
        else:
            return industry

    def insert_one_industry_category(self, data):
        """
        The following function will insert one industry_category in the
        industry_category collections
        :param data:
        :return:
        """
        return self.insert_one(data)

    def update_one_industry_category(self, data, query):
        """
        The following function will update one industry_category in
        industry_category collection based on the given query
        :param data:
        :param upsert:
        :param query:
        :return:
        """
        return self.update_one(data=data, upsert=False, query=query)

    def update_many_industry_category(self, data, query):
        """
        The following function will update many industry_category in
        industry_category collection based on the given query
        :param data:
        :param upsert:
        :param query:
        :return:
        """
        return self.update_many(data=data, upsert=False, query=query)

    def find_all_industry_category(self, sort=None, skip=0, limit=None, **query):
        """
        The following function will give all industry_category for the given set of
        search parameters as keyword arguments
        :param sort:
        :param skip:
        :param limit:
        :param query:
        :return:
        """
        filter_dict = {
            "industry_category_name": 1,
            "industry_category_id": 1,
            "description": 1,
            "is_deleted": 1,
            "upload_icon": 1,
        }
        response = self.find(filter_dict=filter_dict, sort=sort, skip=skip, limit=limit, query=query)
        if not response:
            return []
        return list(response)

    def find_industry_list(self):
        response = self.find({"is_deleted": False})
        if not response:
            return []
        return list(response)

    def fetch_all_industry_category(self, query):
        """
        The following function will give all industry_category for the given set of
        search parameters as keyword arguments
        :param query:
        :return:
        """
        response = self.find(query=query)
        return list(response) if response else []
