from typing import Dict, List, Optional

from pydantic import BaseModel

from scripts.constants.db_constants import DatabaseNames, CollectionNames
from scripts.db.redis_connection import source_space_db
from scripts.utils.mongo_utils import MongoCollectionBaseClass


class AssetDetailSchema(BaseModel):
    """
    This is the Schema for the Mongo DB Collection.
    All datastore and general responses will be following the schema.
    """

    asset_model_id: Optional[str] = ""
    allow_editing: Optional[bool] = True
    asset_description: Optional[str] = ""
    asset_version: Optional[str] = ""
    asset_model_type: Optional[str] = ""
    asset_model_icon: Optional[str] = ""
    parameters: Optional[Dict] = {}
    parameters_new: Optional[Dict] = {}
    processes: Optional[list] = []
    device_models: Optional[List] = []
    events: Optional[List] = []
    resources: Optional[Dict] = {}
    others: Optional[Dict] = {}


class AssetDetail(MongoCollectionBaseClass):
    def __init__(self, mongo_client, project_id=None):
        super().__init__(
            mongo_client, database=DatabaseNames.ilens_asset_model, collection=CollectionNames.asset_model_details, space_db=source_space_db
        )
        self.space_id = project_id

    @property
    def key_project_id(self):
        return "project_id"

    @property
    def key_asset_model_id(self):
        return "asset_model_id"

    @property
    def key_asset_version(self):
        return "asset_version"

    def find_asset_detail_by_param(self, **query):
        asset_list = self.find(query)
        return asset_list

    def find_asset_detail_by_id(self, project_id, asset_id=None, asset_version=None, asset_name=None, filter_dict=None):
        query = {}
        query.update({self.key_project_id: project_id})
        if asset_id:
            query.update({self.key_asset_model_id: asset_id})
        if asset_version:
            query.update({self.key_asset_version: asset_version})
        if asset_name:
            query.update({"asset_model_name": asset_name})
        asset_list = self.find_one(query=query, filter_dict=filter_dict)
        if asset_list:
            return asset_list
        return {}

    def find_assets(self, query):
        all_assets = self.find(query=query)
        if all_assets:
            return list(all_assets)
        return []

    def get_highest_asset_model_version(self, asset_model_name, project_id):
        query = [
            {"$match": {"asset_model_name": asset_model_name, "project_id": project_id}},
            {
                "$group": {
                    "_id": "$asset_model_id",
                    "highestVersion": {"$max": "$asset_version"},
                }
            },
        ]
        res = list(self.aggregate(query))
        if res:
            return res[0].get("highestVersion", "0.0"), res[0].get("_id", "")
        else:
            return "0.0", ""

    def insert_one_asset_detail(self, data):
        """
        The following function will insert one asset in the
        asset_list collections
        :param self:
        :param data:
        :return:
        """
        insert_data = data
        return self.insert_one(insert_data)

    def delete_one_asset_detail(self, asset_id, asset_version):
        query = {}
        if bool(asset_id):
            query.update({self.key_asset_model_id: asset_id})
        if bool(asset_version):
            query.update({self.key_asset_version: asset_version})
        if query:
            return self.delete_one(query)
        else:
            return False

    def delete_one_asset_rule(self, project_id, asset_id, asset_version, rule_engine_id):
        query = {
            self.key_project_id: project_id,
            self.key_asset_model_id: asset_id,
            self.key_asset_version: asset_version,
            "rules.rule_engine_id": rule_engine_id,
        }
        return self.delete_one(query)

    def update_review_status(self, asset_model_id, asset_version, action, project_id, user_id):
        query = {
            self.key_asset_model_id: asset_model_id,
            self.key_asset_version: asset_version,
            self.key_project_id: project_id,
        }
        return self.update_one(query=query, data={"status": action, "action_user": user_id}, upsert=False)

    def update_asset_detail(self, asset_id, data, asset_version=None, project_id=None, upsert=False):
        query = {self.key_asset_model_id: asset_id}
        if asset_version:
            query.update({self.key_asset_version: asset_version})
        if project_id:
            query.update({self.key_project_id: project_id})
        return self.update_one(query=query, data=data, upsert=upsert)

    def update_many_asset_detail(self, asset_id, data, asset_version=None, project_id=None, upsert=False):
        query = {self.key_asset_model_id: asset_id}
        if asset_version:
            query.update({self.key_asset_version: asset_version})
        if project_id:
            query.update({self.key_project_id: project_id})
        return self.update_many(query=query, data=data, upsert=upsert)

    def aggregate_asset_detail(self, filter_list: List):
        if filter_list:
            return self.aggregate(pipelines=filter_list)

    def match_asset_des(self, obj_req):
        query = [
            {"$match": {"asset_description": {"$regex": obj_req, "$options": "i"}}},
            {"$project": {"asset_description": 1, "asset_model_icon": 1, "_id": 0}},
        ]

        return list(self.aggregate(pipelines=query))
