import re
from copy import deepcopy

import bcrypt
from ut_mongo_util import mongo_client

from scripts.config.default_meta_catalog_constants import DEFAULT_USER
from scripts.db.user import User
from scripts.errors import CustomError
from scripts.logging import logger


class UserCreation:
    def __init__(self):
        """
        The __init__ function is called when the class is instantiated.
        It sets up the instance of the class, and defines all of its attributes.


        :param self: Represent the instance of the class
        :param : Pass the mongo client to the class
        :return: The following:
        """
        self.user_conn = User(mongo_client=mongo_client)

    @staticmethod
    def validate_password_strength(password):
        """
        This method is to validate the password strength
        """
        try:
            logger.info("Validate password strength")

            conditions = [
                len(password) > 7,
                len(password) < 65,
                re.search("[a-z]", password) is not None,
                re.search(r"\d", password) is not None,
                re.search("[A-Z]", password) is not None,
                re.search("[!@#$%^&*]", password) is not None,
                not re.search("\\s", password),
            ]

            password_validation_status = all(conditions)
        except Exception as e:
            logger.error(f"Error occurred while validating the password strength : {str(e)}")
            password_validation_status = False

        return password_validation_status

    def encrypt_password(self, password):
        # Decrypt encrypted password
        if not self.validate_password_strength(password):
            message = (
                "Password should contain minimum of 8 characters with at least a symbol, "
                "one upper and one lower case letters and a number"
            )
            raise CustomError(message)
        hash_pass = bcrypt.hashpw(password.encode("utf-8"), bcrypt.gensalt())
        if isinstance(hash_pass, bytes):
            hash_pass = hash_pass.decode()
        return hash_pass

    def create_default_user(self, password):
        try:
            enc_password = self.encrypt_password(password)
            admin_user = deepcopy(DEFAULT_USER)
            admin_user.update({"password": enc_password})
            user_record = self.user_conn.find_user_by_space_id(
                user_id=admin_user.get("user_id"), space_id=admin_user.get("space_id")
            )
            if not user_record:
                self.user_conn.insert_one_user(admin_user)
        except Exception as e:
            logger.error(f"Error creating public default data {str(e)}")
