from typing import Dict, Optional

from pydantic import BaseModel
from scripts.utils.mongo_utils import MongoCollectionBaseClass


class WorkSpacesSchema(BaseModel):
    """
    This is the Schema for the Mongo DB Collection.
    All datastore and general responses will be following the schema.
    """

    space_id: Optional[str] = ""
    space_name: Optional[str] = ""
    space_type: Optional[str] = ""
    meta: Optional[Dict] = {}
    user_id: Optional[str] = ""
    source_meta: Optional[Dict] = {}
    access_token: Optional[str] = ""
    catalog_url: Optional[str] = ""


class WorkSpaces(MongoCollectionBaseClass):
    def __init__(self, mongo_client):
        super().__init__(mongo_client, database="catalog_meta_dub", collection="workspaces")

    @property
    def key_space_id(self):
        return "space_id"

    @property
    def key_space_name(self):
        return "space_name"

    def find_space(self, space_id=None, space_name=None, filter_dict=None):
        """
        The following function will give one record for a given set of
        search parameters as keyword arguments
        :param space_name:
        :param filter_dict:
        :param space_id:
        :return:
        """
        query = {}
        if space_id:
            query.update({self.key_space_id: space_id})
        if space_name:
            query.update({self.key_space_name: space_name})
        record = self.find_one(query=query, filter_dict=filter_dict)
        if not record:
            return {}
        return WorkSpacesSchema(**record).dict()

    def find_space_by_query(self, query, filter_dict=None):
        record = self.find(query=query, filter_dict=filter_dict)
        if record:
            return record
        return []

    def fetch_space_details(self):
        query = {}
        filter_dict = {self.key_space_id: 1, "_id": 0, self.key_space_name: 1}
        records = self.find(query=query, filter_dict=filter_dict)
        if records:
            space_name_mapp = {}
            for record in records:
                space_name_mapp[record.get(self.key_space_id)] = record.get(self.key_space_name)
            return space_name_mapp
        return {}

    def insert_one_space(self, data):
        """
        The following function will insert one space in the
        customer_spaces collections
        :param self:
        :param data:
        :return:
        """
        return self.insert_one(data)

    def delete_one_space(self, space_id):
        if space_id:
            query = {self.key_space_id: space_id}
            return self.delete_one(query)
        else:
            return False

    def get_space_data_by_aggregate(self, query: list):
        return list(self.aggregate(pipelines=query))

    def update_one_space(self, data, space_id, upsert=False):
        """
        The following function will update one step in
        steps collection based on the given query
        :param data:
        :param upsert:
        :param space_id:
        :return:
        """
        query = {"space_id": space_id}
        response = self.update_one(data=data, upsert=upsert, query=query)
        return response

    def delete_workspaces(self, space_id_list):
        query = {self.key_space_id: {"$in": space_id_list}}
        response = self.delete_many(query)
        return response.deleted_count
