import re
from typing import Any, Dict, List, Optional, Union

from pydantic import BaseModel
from ut_mongo_util import CollectionBaseClass
from scripts.utils.decryption_util import MongoDataEncryption
from scripts.utils.mongo_utils import MongoCollectionBaseClass as UtilsMongoCollection


class UserCollectionKeys:
    KEY_LANGUAGE = "language"
    KEY_NAME = "name"
    KEY_USER_ID = "user_id"
    KEY_SPACE_ID = "space_id"
    KEY_USERNAME = "username"
    KEY_USER_ROLE = "userrole"
    KEY_EMAIL = "email"


class UserSchema(BaseModel):
    name: Optional[str] = ""
    space_id: Optional[str] = ""
    username: Optional[str] = ""
    password: Optional[str] = ""
    email: Optional[Any] = None
    phonenumber: Optional[Any] = None
    userrole: Optional[List[str]] = None
    user_type: Optional[str] = ""
    user_id: Optional[str] = ""
    created_by: Optional[str] = ""
    encryption_salt: Optional[Dict] = {}
    passwordReset: Optional[Dict] = {}
    failed_attempts: Optional[int] = 0
    is_user_locked: Optional[bool] = False
    last_failed_login: Optional[int] = 0
    last_logged_in: Optional[int] = 0
    last_failed_attempt: Optional[str] = ""
    expires_on: Optional[str] = ""
    disable_user: Optional[bool] = False
    default_user: Optional[bool] = False
    created_on: Optional[int] = 0
    updated_by: Optional[str] = ""
    updated_on: Optional[int] = 0
    secret: Optional[str] = ""
    password_added_on: Optional[int] = 0
    default_space: Optional[str] = ""
    fixed_delay: Optional[int] = 0
    variable_delay: Optional[int] = 0


class User(CollectionBaseClass):
    def __init__(self, mongo_client, space_id=None):
        super().__init__(mongo_client, database="catalog_meta_dub", collection="user")
        self.space_id = space_id
        self.key_user_id = UserCollectionKeys.KEY_USER_ID
        self.key_space_id = UserCollectionKeys.KEY_SPACE_ID
        self.key_username = UserCollectionKeys.KEY_USERNAME
        self.key_email = UserCollectionKeys.KEY_EMAIL
        self.find_decrypted = UtilsMongoCollection.find_decrypted.__get__(self, UtilsMongoCollection)
        self.get_decrypted_records = UtilsMongoCollection.get_decrypted_records.__get__(self, UtilsMongoCollection)
        self.data_encryption = MongoDataEncryption()

    def update_user(self, query, data):
        """
        The following function will update target details in rule_targets collections
        :param self:
        :param data:
        :return:
        """
        return self.update_one(query=query, data=data, upsert=True)

    def insert_one_user(self, data):
        """
        The following function will insert one user in the
        user collections
        :param self:
        :param data:
        :return:
        """
        data = self.data_encryption.encrypt_data(data, collection_name="user")
        return self.insert_one(data)

    def find_user(self, space_id, user_id=None, username=None, email=None, filter_dict=None):
        query = {}
        if user_id:
            query[self.key_user_id] = user_id
        if username:
            query[self.key_username] = username
        if email:
            query[self.key_email] = re.compile(email, re.IGNORECASE)
            query[self.key_email] = email
        user = self.find_decrypted(query=query, filter_dict=filter_dict)
        if user:
            return UserSchema(**user)
        return user

    @staticmethod
    def get_users_list(space_id=None):
        query_json = [
            {
                "$group": {
                    "_id": None,
                    "data": {"$push": {"k": {"$ifNull": ["$user_id", ""]}, "v": {"$ifNull": ["$username", ""]}}},
                }
            },
            {"$replaceRoot": {"newRoot": {"$arrayToObject": "$data"}}},
        ]
        if space_id:
            query_json.insert(0, {"$match": {"space_id": space_id}})
        return query_json

    def users_list_by_aggregate(self, query: list):
        return self.aggregate(pipelines=query)

    def find_user_by_space_id(self, user_id, space_id):
        user = self.find_one(query={self.key_user_id: user_id, self.key_space_id: space_id})
        if user:
            return dict(user)
        return user

    def get_all_users(self, filter_dict=None, sort=None, skip=0, limit=None, **query):
        users = self.find(filter_dict=filter_dict, sort=sort, skip=skip, limit=limit, query=query)
        if users:
            return list(users)
        return []

    def find_user_role_for_user_id(self, user_id, space_id):
        query = {"user_id": user_id, "space_id": space_id}
        filter_dict = {"userrole": 1, "_id": 0}
        return self.find_one(query=query, filter_dict=filter_dict)

    def find_base_user(self, space_id=None, user_id=None, username=None, email=None, filter_dict=None):
        query = {}
        if space_id:
            query[self.key_space_id] = space_id
        if user_id:
            query[self.key_user_id] = user_id
        if username:
            query[self.key_username] = username
        if email:
            query[self.key_email] = re.compile(email, re.IGNORECASE)
        if not (user := self.find_decrypted(query=query, filter_dict=filter_dict)):
            return user
        try:
            return UserSchema(**user)
        except Exception:
            return user

    def find_by_space(
        self,
        projections=None,
        sort=None,
        query_dict=None,
        limit=None,
        skip=0,
        **filters,
    ) -> Union[Any, None]:
        query = {}
        if query_dict:
            query |= query_dict
        if filters:
            query.update(filters)
        records = self.find(query, projections, sort=sort, limit=limit, skip=skip)
        if records:
            records = self.get_decrypted_records(records)
        return list(records) if records else []

    def delete_one_user(self, user_id, space_id):
        return self.delete_one(query={self.key_user_id: user_id, self.key_space_id: space_id})

    def update_one_user(self, query, data):
        """
        The following function will insert one user in the
        user collections
        :param self:
        :param data:
        :return:
        """
        data = self.data_encryption.encrypt_data(data, collection_name="user")
        return self.update_one(query=query, data=data, upsert=True)

    def get_data_by_aggregate(self, query_json: list):
        if aggregate_data := list(self.aggregate(query_json)):
            aggregate_data = self.get_decrypted_records(aggregate_data)

            return aggregate_data
        return []

    def find_by_aggregate(self, query_json: list):
        if user_by_aggregate := list(self.aggregate(query_json)):
            return user_by_aggregate
        return []

    def distinct_user(self, query_key, filter_json):
        query = {self.key_user_id: filter_json}
        return self.distinct(query_key=query_key, filter_json=query)

    def find_user_by_param(self, **query):
        user = self.get_decrypted_records(self.find(query))

        if not bool(user):
            user = []
        return user
