import json
import logging
import time
from copy import deepcopy
from ut_mongo_util import mongo_client

from scripts.config.default_meta_catalog_constants import DEFAULT_SPACE
from scripts.db.user_space import UserSpace
from scripts.db.workspaces import WorkSpaces
from scripts.db.redis_connection import space_db
from scripts.errors import WorkspaceNameExistError
from scripts.schema import CreateWorkspace


class WorkspaceCreation:
    def __init__(self):
        """
        The __init__ function is called when the class is instantiated.
        It sets up the instance of the class, and defines all of its attributes.


        :param self: Represent the instance of the class
        :param : Pass the mongo client to the class
        :return: The following:
        """
        self.workspace_conn = WorkSpaces(mongo_client=mongo_client)
        self.user_space_conn = UserSpace()

    def validate_name_catalog(self, workspace_name, space_id):
        try:
            existing_space_details = self.workspace_conn.find_space(space_name=workspace_name, space_id=space_id)
            return existing_space_details

        except Exception as e:
            logging.error(f"Error occurred in the validate name in catalog {str(e)}")

    @staticmethod
    def set_or_update_redis(redis_client, add_prefix_to_database, space_id):
        """
        Checks if a key exists in Redis and inserts or updates it with the given value.
        Dynamically builds the value based on `add_prefix_to_database` and `space_id`.

        :param redis_client: Redis client instance.
        :param add_prefix_to_database: Boolean flag indicating if the prefix should be added.
        :param space_id: The space_id used as the prefix for the database.
        """
        key = space_id

        # Build the dynamic source_meta dictionary
        source_meta = {
            "add_prefix_to_database": add_prefix_to_database,
            "prefix": space_id if add_prefix_to_database else "",
        }

        # Prepare the JSON object to store in Redis
        value_json = json.dumps({"source_meta": source_meta})

        try:
            # Insert or update the key in Redis
            redis_client.set(key, value_json)
            logging.info(f"Key '{key}' has been set or updated successfully.")
        except Exception as e:
            logging.error(f"Error occurred while setting/updating key '{key}': {e}")
            raise

    def global_catalog_workspace_creation(self):
        try:
            user_id = "user_097"
            data = deepcopy(
                CreateWorkspace(
                    space_id="space_099", space_name="Central Workspace", space_type="public", user_id=user_id
                )
            )
            existing_space_details = self.validate_name_catalog(workspace_name=data.space_name, space_id=data.space_id)
            if existing_space_details:
                logging.debug(f"It is already existing space_name is {str(data.space_name)}")
                return {"space_id": data.space_id}

            data.meta.update({"updated_at": int(time.time() * 1000), "created_by": user_id, "updated_by": user_id})
            count = self.workspace_conn.update_one_space(data.dict(), data.space_id, upsert=True)
            logging.debug(f"Updated Count {str(count)} ")
            self.set_or_update_redis(space_db, add_prefix_to_database=False, space_id=data.space_id)
            self.user_space_conn.update_one_user_space(DEFAULT_SPACE, DEFAULT_SPACE.get("user_id"),
                                              DEFAULT_SPACE.get("space_id"))

            return {"space_id": data.space_id}
        except WorkspaceNameExistError:
            raise WorkspaceNameExistError
        except Exception as e:
            logging.error(f"Error occurred in the global catalog creation due to {str(e)}")